#!/bin/ksh
#
#######################################################################
# Copyright (c) 1997-2004 Sun Microsystems, Inc. All Rights Reserved.
#
# SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
# THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
# TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
# PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
# ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
# DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
#
## Source:               s9_preupgrade_prepare.sh
## Application Name:     ITSM S9 Migration Tools
#
## Author:        CH
## Date Created:  28-OCT-2002
#
## General Description(Purpose): 
#       This script prepares the environment/system for premigration
#  module. Called from s9_premigration.sh module and will
#	1. Validate Target OS Level
#	2. Validate Target OS Packages
#	3. Validate memory requirements
#	4. Validate memory architecture
#	5. Validate processor
#	6. Validate required LU packages
#	7. Validate required LU patches
#	8. Give summary report
#
## Usage:   Called from s9_premigration.sh.
#	    s9_premigration_prepare.sh
#
## Required Parameters:   N/A
#
## Optional Parameters:   N/A
## Return Codes:          0 or 1
#
## Possible Points of Failure: This script could fail if the
#  steps associated in documents are not followed.
#
## Modification History:
#  -------------------------------------------------------------
#  Date           Name                     Comments
#  -------------------------------------------------------------
#  28-OCT-2002    CH            Initial Creation
#  03-DEC-2002    CH            Added processor check
#  13-DEC-2002    SG		return only 1 value for memory size
#  18-DEC-2002 	  SG		Enhanced error messages
#  13-JAN-2003    CH            Error logging RFE's
#  16-JAN-2003    CH            Unix cmd paths
#  09-MAY-2003    UM	        Added fuction disp_vm_info to capture
#                               volume manager info based on root
#######################################################################

l_pkg_dir="/opt"
l_g_env_file="${l_pkg_dir}/${p_pkg_name}/etc/set_prepare_env"
l_s_env_file="${l_pkg_dir}/${p_pkg_name}/etc/set_migration_env"
l_d_env_file="${l_pkg_dir}/${p_pkg_name}/etc/s9_`hostname`_env"
l_script_name=`basename $0 .sh`
l_logdir="/var/sadm/system/logs"
PATH="/usr/bin:/usr/sbin:${PATH}:${l_pkg_dir}/${p_pkg_name}/bin"
DATE="/usr/bin/date +%Y%m%d_%T"

#Verify for existence

if [ -x ${l_s_env_file} ]
then
    . ${l_s_env_file}
else
    echo "\n------------------------------------------------------------------------"
    echo "[`${DATE}`] ERROR: ${l_s_env_file} does not exist, exiting"
    exit 1
fi

if [ -x ${l_g_env_file} ]
then
    . ${l_g_env_file}
else
    echo "\n------------------------------------------------------------------------"
    echo "[`${DATE}`] ERROR: ${l_g_env_file} does not exist, exiting"
    exit 1
fi

Print_MSG START ${l_script_name}


check_error()
{
  l_c=`expr $1 + 0`
  l_s=$2
  if [ ${l_c} -ne "0" ]
  then
      Print_MSG ERROR ${l_s}
      exit 1
  fi
}

set_var()
{

  echo "export p_lupkginstall=\"${l_luinstallstring}\""  >> ${l_d_env_file}
  echo "export p_lupkgremove=\"${l_luremovestring}\"" >> ${l_d_env_file}
  echo "export p_lupatch=\"${l_lupatchstring}\""  >> ${l_d_env_file}
  echo "export p_oslevel=\"${l_oslevelstring}\"" >> ${l_d_env_file}
  echo "export p_ospackage=\"${l_ospackagestring}\"" >> ${l_d_env_file}
  echo "export p_arch=\"${l_archstring}\"" >> ${l_d_env_file}
  echo "export p_memory=${l_memorystring}" >> ${l_d_env_file}
  echo "export p_proc_type=\"${l_procstring}\"" >> ${l_d_env_file}
  echo "export p_alt_be_name=\"${p_alt_be_default}\"" >> ${l_d_env_file}
}

disp_vm_info()
{
  l_vm_name=$1
  echo ""
  echo "Volume Manager Information"
  echo "=========================="
  /usr/bin/pkginfo -x ${l_vm_name}
  echo ""
}

# Collect System Information

  Print_MSG INFO "Collecting System Information" 

  ${p_bin_dir}/s9_parfacts.sh >/dev/null 2>&1 </dev/null

  check_error $? "Collecting system information"
  sleep 2


#Step - 1 Validate Target OS Level

  l_step_name="VALIDATE_OS_LEVEL"
  l_os_level=`/usr/bin/uname -r`
  l_os_check="ok"
  l_valid=""
  typeset -L15 l_os_type

  Print_MSG BEGIN ${l_step_name}


  echo ""
  echo "Current OS Level      Required for Upgrade"
  echo "=================     ===================="

  for l_this_os in ${l_os_version}
  do
      if [[ ${l_this_os} = ${l_os_level} ]]
      then
          l_os_type=${l_this_os}
          echo "${l_os_type}       Pass - Valid Level"
          l_valid="ok"
       fi
  done

  if [[ ${l_valid} != "ok" ]]
  then
      echo "${l_os_type}        Fail - Invalid"
      l_os_check="error"
  fi

  l_oslevelstring="${l_os_level}"
  echo ""    
  Print_MSG END ${l_step_name}
  sleep 2

#Step - 2 Validate Target OS Packages
  
  l_step_name="VALIDATE_OS_PKGS"
  l_os_pkg_check="ok"
  l_os_pkg="l_ospkg`/usr/bin/uname -r | /usr/bin/tr '.' '_'`"
  l_pkg_file="${l_logdir}/s9_parfacts_pkg.out"
  typeset -L15 l_req_pkg

  Print_MSG BEGIN ${l_step_name}

  l_packageList=`eval "echo \\${$l_os_pkg}"`

  echo ""
  echo "OS Level Packages     Required for LU V2.0"
  echo "=================     ===================="

  for l_this_pkg in $l_packageList
  do
      l_req_pkg=${l_this_pkg}
      l_packageline=`/usr/bin/grep "${l_req_pkg}" ${l_pkg_file} | /usr/bin/tail -1`

      if [[ ${l_packageline}  = "" ]]
      then
          l_foundone=0
      else
          l_foundone=1
          l_foundpkg=`echo ${l_packageline} | /usr/bin/awk '{print $2}'`   
      fi

      ## Fix for bzip package required for Sol8

      if [ ${l_foundone} -eq 0 -a ${l_this_pkg} = "SUNWbzip" ]
      then
          l_bzip_fix=${l_this_pkg}
      elif [ ${l_foundone} -eq 0 ]
      then
       echo "${l_req_pkg}       Not Installed"
       l_os_pkg_check="error"
       l_ospackagestring="$l_ospackagestring${l_this_pkg}  "
      else
          echo "${l_req_pkg}       Installed"
      fi

  done
  echo ""

  l_root_dev=`/usr/bin/nawk ' { if($2=="\/") print $1 }' /etc/mnttab | awk -F\/ ' { print $3} '`

  [ "${l_root_dev}" = "vx" ] && l_vm_name=`/usr/bin/grep "vxvm" ${l_pkg_file} | /usr/bin/tail -1 | /usr/bin/awk ' { print $2 }'`
  [ "${l_root_dev}" = "md" ] && l_vm_name=`/usr/bin/grep "SUNWmd" ${l_pkg_file} | /usr/bin/tail -1 | /usr/bin/awk ' { print $2 }'`
  [ -n "${l_vm_name}" ] && disp_vm_info ${l_vm_name}

  Print_MSG END ${l_step_name}
  sleep 2

#Step - 3 Validate memory requirements

  l_step_name="VALIDATE_MEMORY"
  l_memory_check="ok"
  l_memory_num=""
  typeset -L15 l_memory_display

  Print_MSG BEGIN ${l_step_name}

  echo ""
  echo "HW Level Memory       Required for S9 OE"
  echo "=================     ===================="

  l_memory_display=`/usr/bin/grep "Memory size" ${l_logdir}/s9_parfacts.out | /usr/bin/awk '{print $3, $4}' | \
  /usr/bin/tail -1`
  l_memory_num=`/usr/bin/grep "Memory size" ${l_logdir}/s9_parfacts.out | /usr/bin/awk '{print $3}'| \
  /usr/bin/cut -f1 -d 'M' | /usr/bin/tail -1`        


  if [[ ${l_memory_num} -ge ${l_memory_threshold} ]]
  then
      echo "${l_memory_display}       Pass - Valid Level"
  else
      echo "${l_memory_display}       Fail - Invalid"
      l_memory_check="error"
  fi

  l_memorystring="${l_memory_num}"
  echo ""

  Print_MSG END ${l_step_name}
  sleep 2

#Step - 4 Validate memory architecture

  l_step_name="VALIDATE_ARCH"
  l_hw_check="ok"
  l_valid=""
  l_hw_arch=`/usr/bin/uname -m`
  typeset -L15 l_hw_type

  Print_MSG BEGIN ${l_step_name}

  echo ""
  echo "HW Level ARCH         Required for LU V2.0"
  echo "=================     ===================="

  for l_this_hw in ${l_hw_version}
  do
      if [[ ${l_this_hw} = ${l_hw_arch} ]]
      then
          l_hw_type=${l_this_hw}
          echo "${l_hw_type}       Pass - Valid Level"
          l_valid="ok"
      fi
  done

  if [[ ${l_valid} != "ok" ]]
  then
      echo "${l_hw_type}        Fail - Invalid"
      l_hw_check="error" 
  fi

  l_archstring="${l_hw_arch}"
  echo "" 
  Print_MSG END ${l_step_name}
  sleep 2

#Step - 5 Validate processor

  l_step_name="VALIDATE_PROCESSOR"
  l_proc_check="ok"
  l_valid=""
  l_proc=`/usr/bin/uname -p`
  typeset -L15 l_proc_type

  Print_MSG BEGIN ${l_step_name}

  echo ""
  echo "Processor Type        Required for V1.1"
  echo "=================     ===================="

  for l_this_proc in ${l_proc_version}
  do
      if [[ ${l_this_proc} = ${l_proc} ]]
      then
          l_proc_type=${l_this_proc}
          echo "${l_proc_type}       Pass - Valid Level"
          l_valid="ok"
      fi
  done

  if [[ ${l_valid} != "ok" ]]
  then
      echo "${l_proc_type}        Fail - Invalid"
      l_proc_check="error"
  fi

  l_procstring="$l_procstring${l_proc}"
  echo ""
  Print_MSG END ${l_step_name}
  sleep 2

#Step - 6 Validate required LU packages

  l_step_name="VALIDATE_LU_PKGS"
  l_lu_pkg_check="ok"
  typeset -L15 l_lu_pkg
  
  Print_MSG BEGIN ${l_step_name}

  echo ""
  echo "LU Packages           Required for LU V2.0"
  echo "=================     ===================="


  ## Fix for bzip package required for Sol8

  if [[ ! -z ${l_bzip_fix} ]]
  then
     l_lu_pkg=${l_bzip_fix}
     echo "${l_lu_pkg}       Sol8 only - Not Installed"
     l_luinstallstring="$l_luinstallstring${l_bzip_fix}  "
     l_lu_pkg_check="auto-install"
  fi


  for l_software_pkg in $l_lu_pkg_list
  do
      l_lu_pkg=${l_software_pkg}
      l_lupackageline=`/usr/bin/grep ${l_lu_pkg} ${l_pkg_file} | /usr/bin/tail -1`

      if [[ -z "${l_lupackageline}" ]]
      then
          l_lufoundone=0
      else
          l_lufoundone=1
          l_lufoundpkgrev=`/bin/pkgparam ${l_software_pkg} VERSION 2>/dev/null | /usr/bin/cut -f1 -d ',' 2>/dev/null`
      fi

      if [[ ${l_lufoundone} -eq 0 ]]
      then
          echo "${l_lu_pkg}       Not Installed"
          l_luinstallstring="$l_luinstallstring${l_software_pkg}  "
          l_lu_pkg_check="auto-install"
      elif [ "${l_lu_pkg_version}" != "${l_lufoundpkgrev}" ]
      then
          if [ -z "${l_lufoundpkgrev}" ]
          then
              echo "${l_lu_pkg}       Not Installed"
          else
              echo "${l_lu_pkg}       Fail - Version (${l_lufoundpkgrev}) too low Upgrade Required!"
              l_luremovestring="$l_luremovestring${l_software_pkg}  "
          fi
          l_luinstallstring="$l_luinstallstring${l_software_pkg}  "
          l_lu_pkg_check="auto-install"
      else
          echo "${l_lu_pkg}       Installed"
      fi

   done
   echo "" 
   Print_MSG END ${l_step_name}
   sleep 2


#Step - 7 Validate required LU patches

  l_step_name="VALIDATE_LU_PATCHES"
  l_lu_patch_check="ok"
  l_patch_level="l_lupatch`/usr/bin/uname -r | /usr/bin/tr '.' '_'`"
  l_patchList=`eval "echo \\${$l_patch_level}"`
  l_patch_file="${l_logdir}/s9_parfacts_patch.out"
  typeset -L15 l_patchform

  Print_MSG BEGIN ${l_step_name}

  /usr/bin/showrev -p | /usr/bin/awk -F: '{print $1, $2, $3}'|\
            /usr/bin/sed -e 's/Packages//'\
            -e 's/Obsoletes $//' |\
            /usr/bin/sort +1 -2 -n > $l_patch_file


  echo ""
  echo "LU Patches            Required for LU V2.0"
  echo "=================     ===================="

  for l_lu_patch in $l_patchList
  do
      l_patchform=${l_lu_patch%%}
      l_checkpatch=${l_patchform%%-*}
      l_checkpatchrev=${l_patchform#*-}

      l_lupatchline=`/usr/bin/grep ${l_checkpatch} ${l_patch_file} | /usr/bin/tail -1`

      if [[ -z "${l_lupatchline}" ]]
      then
          l_lufoundpatch=0
      else
          l_lufoundpatch=1
         _l_lupatch=`echo ${l_lupatchline} | /usr/bin/awk '{print $2}'`
          l_lufoundpatch=${_l_lupatch%%-*}
          l_lufoundpatchrev=${_l_lupatch#*-}
      fi

      if [ ${l_lufoundpatch} -eq 0 ]
      then
          echo "${l_patchform}       Not Installed"
          l_lupatchstring="$l_lupatchstring${l_lu_patch}  "
          l_lu_patch_check="auto-install"

      elif [ $l_checkpatch -ne  $l_lufoundpatch ]
      then
          echo "${l_patchform}       Pass - Obsoleted"
      elif [ $l_checkpatchrev -gt $l_lufoundpatchrev ]
      then
          echo "${l_patchform}       Fail - Rev (-${l_lufoundpatchrev}) too low Upgrade Required!"
          l_lupatchstring="$l_lupatchstring${l_lu_patch}  "
          l_lu_patch_check="auto-install"
      else
          echo "${l_patchform}       Installed"
      fi
  done
  echo "" 
  Print_MSG END ${l_step_name}
  sleep 2

# Validation Summary

  Print_MSG INFO "Creating Summary Report"

  echo ""
  echo "Valdation status S9 OS Req. . . . . . . . .[${l_os_check}]"
  echo "Valdation status S9 OS Package Req. . . . .[${l_os_pkg_check}]"
  echo "Valdation status S9 Memory Req. . . . . . .[${l_memory_check}]"
  echo "Valdation status S9 HW Req. . . . . . . . .[${l_hw_check}]"
  echo "Valdation status S9 Processor Type. . . . .[${l_proc_check}]"
  echo "Valdation status LiveUpgrade Packages . . .[${l_lu_pkg_check}]"
  echo "Valdation status LiveUpgrade Patches. . . .[${l_lu_patch_check}]"
  echo ""
  sleep 2

  if [ ${l_os_check} = "error" ]
  then
     Print_MSG ERROR "Unsupported version of Solaris (${l_os_level}) detected"
     echo ""
     echo "  Supported versions of Solaris: ${l_os_version}"
     echo "  Please upgrade to a supported version before attempting S9 Migration process"
     echo ""
     echo "  After correcting the error, please refer to the S9 Migration Process Guide document"
     echo "  for instructions on re-running the s9_premigration step"
     echo ""
     sleep 2
     exit 1
  fi
  if [ ${l_os_pkg_check} = "error" ]
  then
     Print_MSG ERROR "Required OS Level Package (${l_this_pkg}) not detected"
     echo ""
     echo "  This version of Solaris (${l_os_level}) requires OS Packages"
     echo "  ${l_packageList}"
     echo "  Please install Package ${l_this_pkg} before attempting S9 Migration process"
     echo ""
     echo "  After correcting the error, please refer to the S9 Migration Process Guide document"
     echo "  for instructions on re-running the s9_premigration step"
     echo ""
     sleep 2
     exit 1
  fi
  if [ ${l_memory_check} = "error" ]
  then
     Print_MSG ERROR "Required minimum memory (${l_memory_threshold}Mb) not available"
     echo ""
     echo "  Available Memory (${l_memory_display}) less than required minimum"
     echo "  Please address this shortfall before attempting S9 Migration process"
     echo ""
     echo "  After correcting the error, please refer to the S9 Migration Process Guide document"
     echo "  for instructions on re-running the s9_premigration step"
     echo ""
     sleep 2
     exit 1
  fi
  if [ ${l_hw_check} = "error" ]
  then
     Print_MSG ERROR "Unsupported version of HW architecture (${l_hw_arch}) detected"
     echo ""
     echo "  Supported HW architectures : ${l_hw_version}"
     echo "  Please address this issue before attempting S9 Migration process"
     echo ""
     echo "  After correcting the error, please refer to the S9 Migration Process Guide document"
     echo "  for instructions on re-running the s9_premigration step"
     echo ""
     sleep 2
     exit 1
  fi
  if [ ${l_proc_check} = "error" ]
  then
     Print_MSG ERROR "Unsupported processor type (${l_proc}) detected"
     echo ""
     echo "  Supported Processor Types : ${l_proc_version}"
     echo "  Please address this issue before attempting S9 Migration process"
     echo ""
     echo "  After correcting the error, please refer to the S9 Migration Process Guide document"
     echo "  for instructions on re-running the s9_premigration step"
     echo ""
     sleep 2
     exit 1
  fi

  if [ ${l_lu_pkg_check} = "auto-install" -o ${l_lu_patch_check} = "auto-install" ]
  then
     Print_MSG INFO "Some Live Upgrade Patch/Package(s) will be installed"
     set_var
     sleep 2
     Print_MSG FINISH 
     exit 2

  else
     Print_MSG INFO "Environment Status OK"
     set_var
     sleep 2
  fi

  Print_MSG FINISH 

exit 0
