#!/bin/ksh
#
#######################################################################
# Copyright (c) 1997-2004 Sun Microsystems, Inc. All Rights Reserved.
#
# SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
# THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
# TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
# PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
# ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
# DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
#
## Source:               s9_migration.sh
## Application Name:     IT S9 Migration Tools
#
## Author:        UM
## Date Created:  10-NOV-2002
#
## General Description (Purpose): 
#     This program performs or calls migration tasks for S9 migration.
#     It verifies OS disk requirements and partition table for S9.
#     It creates Alternate Boot Environment for S9 and initializes.
#     Program will install selected Solaris Flash Image from
#     known location to ABE. It also performs or calls steps to complete
#     migration and certain OS file/dir synchronization.
#  
## Usage:   This script will be called from Run_S9_migration_process.sh
#           Run_S9_migration_process.sh s9_migration
#
## Required Parameters:   The script prompts for Parameters with
#                         Default Displayed
## Optional Parameters:   N/A
## Return Codes:          0 or 1
#
## Possible Points of Failure: This script could fail if the 
#  steps associated in documents are not followed.
#
## Modification History:
#  -------------------------------------------------------------
#  Date           Name                     Comments    
#  -------------------------------------------------------------
#  17-JUL-2003    UM                       Fix the sort in accept_flash function
#  12-JUN-2003    UM                       Fix process_vfstab function to
#                                          populate the /var entry
#  07-MAY-2003    UM                       Fix check_duplicate function
#                                          Modified create_smmsp_user function
#                                          to user /usr/bin/true instead of default
#                                          /bin/sh
#  15-APR-2003    UM                       Added a function chk_dump_device
#  04-FEB-2003    UM                       Fix the checking on return code for
#                                          egrep in process_files function,currently
#                                          it raises WARNING if it returns 1 which
#                                          is not correct
#  27-JAN-2003    UM                       Added process_list and process_files
#                                          module to handle special files like
#                                          [aliases,services] in future planning
#                                          to add more files to list
#  22-JAN-2003    UM                       Added set_bit_mode function to set
#                                          bit mode in ABE automatically
#  15-JAN-2003    UM                       Change the logic to process sds stuff
#  14-JAN-2003    UM                       replace "=" with "-eq" while comparing
#                                          with numeric 0
#  13-JAN-2003    UM                       Added function initliaze_files &
#                                          remove_files
#  09-JAN-2003    UM                       Added flash_info function
#  08-JAN-2003    UM                       Added the funtion record_flash
#                                          to make sure the selected flash
#                                          can be used incase of the restart
#                                          of the s9_migration script after
#                                          unconditional failure
#  07-JAN-2003    UM                       Added create_smmsp_user function
#  06-JAN-2003    UM                       Modified the logic to create
#                                          lofs mount points .It will 
#                                          create the lofs directories
#                                          only if mount point is either
#                                          "/" or "/var"
#  24-DEC-2002    UM                       Added module to handle 
#                                          migration of Disk Suite
#                                          databases
#  23-DEC-2002    UM                       1. Corrected the bug in 
#                                             select_rows function.
#                                             Handled scenario when stty
#                                             returns rows=0
#  17-DEC-2002    UM                       1. Added to support fmthard
#                                          2. Added lofs mount point
#                                             creation logic.
#  02-DEC-2002    UM                       Added few new features
#                                          1. Displaying list of 
#                                             available flash image and
#                                             allow user to select
#                                          2. Added creating sysidcfg
#                                             file
#                                          3. Added processing /etc/
#                                             system file
#                                          Corrected bugs
#                                          1. Cleaned up variables
#                                          2. added > /dev/null for 
#                                             lustatus command
#                                          3. added -m for luumount and
#                                             mount_point for lumount  
#  10-NOV-2002    UM                       Initial Creation
#######################################################################

l_script_name=`basename $0 .sh`
l_done_dir="/tmp"
l_pkg_dir="/opt"
l_s_env_file="${l_pkg_dir}/${p_pkg_name}/etc/set_migration_env"
l_chk_part_file="${l_pkg_dir}/${p_pkg_name}/etc/s9_chk_partitions.kl"
l_obs_param_file="${l_pkg_dir}/${p_pkg_name}/etc/s9_system_params"
l_sysidcfg_file="${l_pkg_dir}/${p_pkg_name}/etc/s9_sysidcfg"
l_exclude_dir="${l_pkg_dir}/${p_pkg_name}/etc/exclude_dir"
PATH="/usr/bin:/usr/sbin:${PATH}:${l_pkg_dir}/${p_pkg_name}/bin:/usr/opt/SUNWmd/sbin"
DATE="/usr/bin/date +%Y%m%d_%T"


#Verify for existence

if [ -f ${l_s_env_file} ]
then
   . ${l_s_env_file}
else
    echo "\n------------------------------------------------------------------------"
    echo "[`${DATE}`] ERROR: ${l_s_env_file} does not exist, exiting"
    exit 1
fi

if [ -f ${l_chk_part_file} ]
then
   . ${l_chk_part_file}
else
    echo "\n------------------------------------------------------------------------"
    echo "[`${DATE}`] ERROR: ${l_chk_part_file} does not exist, exiting"
    exit 1
fi

# Start defining modules

read_d_env()
{
  if [ -f "${p_d_env_file}" ]
  then
      . ${p_d_env_file}
  else
       Print_MSG ERROR "Run s9_premigration before running ${l_script_name}"
       exit 1
  fi
}

run_luumount()
{
 l_be=$1
 if [ -f "/usr/sbin/luumount" ]
 then
    /usr/sbin/luumount -m /.alt.${l_be} > /dev/null
    if [ $? -ne 0 ]
    then
        Print_MSG ERROR "could not umount ${l_be}"
        echo "Unmount ${l_be} manually using luumount command"
    fi
 else
    Print_MSG ERROR "luumount not found"
    exit  1
 fi
}


check_error()
{
  l_c=`expr $1 + 0`
  l_s=$2
  if [ ${l_c} -ne 0 ]
  then
      update_status "N"
      Print_MSG ERROR ${l_s}
      if [[ ! -z "${l_step_name}" && "${l_step_name}" = "RUNS9_COPY_FILES" ]]
      then
          run_luumount ${p_alt_be_name}
      fi
      exit 1
  fi
}   

check_done()
{
  l_a=$1
  l_done_flag="N"
  if [ -f "${l_done_dir}/${l_a}" ]
  then 
      l_done_flag="Y"
  fi
}

rewrite_alt_disk()
{
  l_disk=$1
  if [ -z "${p_alt_be_disk}" ]
  then
      echo "export p_alt_be_disk=${l_disk}" >> ${p_d_env_file}
      check_error $? "Failed adding variable"
  else
      /usr/bin/sed \
          -e '/p_alt_be_disk=/d' ${p_d_env_file} > ${p_d_env_file}.tmp
      check_error $? "Failed editing env file"
      echo "p_alt_be_disk=\"${l_disk}\"" >> ${p_d_env_file}.tmp
      check_error $? "Failed editing env file"
      /usr/bin/mv ${p_d_env_file}.tmp ${p_d_env_file}
      check_error $? "Failed updating env file"
  fi
}

accept_disk()
{
  echo "\n================================================="
  echo "Please enter disk to use by ABE in <c0t0d0> format"
  echo "Hit enter to accept default"
  echo "\n<default ${p_alt_be_disk}> : \c"
  read l_response
  Print_RSP ${l_response}

  if [ "${l_response}" = "" ]
  then
      Validate_Disk
      check_error $? "Failed Disk Validation" 
      Gather_VTOC_Info Y
      check_error $? "Failed Gathering Disk Partition Information"
      Verify_Slices
      if [ $? -eq 0 -a ${l_error} -eq 1 ]
      then
          Check_Cbe
          check_error $? "Failed Automated Partition"
      else
          if [ $? -ne 0 ]
          then
              check_error $? "Failed Disk Slice Validation"
          fi
      fi
  else
      if [ "${l_response}" = "\n" ]
      then
          Validate_Disk
          check_error $? "Failed Disk Validation"
          Gather_VTOC_Info Y
          check_error $? "Failed Gathering Disk Partition Information"
          Verify_Slices
          if [ $? -eq 0 -a ${l_error} -eq 1 ]
          then
              Check_Cbe
              check_error $? "Failed Automated Partition"
          else
              if [ $? -ne 0 ]
              then
                  check_error $? "Failed Disk Slice Validation"
              fi
          fi
      else
          if [ "${l_response}" != "" ]
          then
              if [ "${l_response}" != "${p_alt_be_disk}" ]
              then
                  rewrite_alt_disk ${l_response}
              fi
              read_d_env
              Validate_Disk
              check_error $? "Failed Disk Validation"
              Gather_VTOC_Info Y
              check_error $? "Failed Gathering Disk Partition Information"
              Verify_Slices
              if [ $? -eq 0 -a ${l_error} -eq 1 ]
              then
                  Check_Cbe
                  check_error $? "Failed Automated Partition"
              else
                  if [ $? -ne 0 ]
                  then
                      check_error $? "Failed Disk Slice Validation"
                  fi
              fi
          else
              if [[ "${p_alt_be_disk}" = "" && "${l_response}" = "" ]]
              then
                  Print_MSG ERROR "Wrong response"
                  exit 2
              fi
          fi
      fi
   fi
}

select_rows()
{
    l_rows=`/usr/bin/stty | /usr/bin/nawk '
      {
        if(/rows/) {
          x=split($0,p,";");
          for ( i = 1; i < x; i++ ) {
            if(substr(p[i],1,4) == "rows") {
              y=split(p[i],q,"=");
              printf("%d\n",q[2]);
              exit(0);
             }
           }
         }
      }'`
    l_rows=${l_rows:-0}
      l_rows=`expr ${l_rows} + 0`
      if [ ${l_rows} -eq 0 ]
      then
          l_rows=23
      fi
}

desc_flash()
{
 l_file_name=$1
 l_line_no=$2

/usr/bin/head -25 ${p_flash_loc}/${l_file_name}|/usr/bin/sed 's/\=/\ /g'| /usr/bin/nawk \
  -v "fname=${l_file_name}" -v "lno=${l_line_no}" '
 {
    if($1 == "creation_date") {
      l_creation_date = substr($2,1,8);
    }
    else if($1 == "creation_hardware_class") {
      l_hw_class      = $2;
    }
    else if($1 == "creation_release") {
      l_os_release    = $2;
    }
    if($1 == "creation_os_version") {
      l_os_version    = $2;
    }
 }
 END {
  printf("%-5d%-6s%-5s%-19s%-9s%-35s\n" \
  ,lno,l_hw_class,l_os_release,l_os_version,l_creation_date,fname);
 }'
}

disp_flash()
{
 echo "Enter [m/M] for more OR Enter [s/S] to select : \c"
 while true
 do
  read l_choice
  Print_RSP ${l_choice}
  l_choice=`echo ${l_choice}|tr "[a-z]" "[A-Z]"`
  if [[ "${l_choice}" = "M" || "${l_choice}" = "S" ]]
  then
      break
  else
      echo "Enter correct choice : \c"
      continue
  fi
 done
}

select_flash()
{
 if [ "${l_choice}" = "S" ]
 then
     let l_sel=0
     echo "Enter number : \c"
     while true
     do
      read l_sel
      Print_RSP ${l_sel}
      expr ${l_sel} + 0 > /dev/null 2>&1
      if [ $? -ne 0 ]
      then
          echo "Enter only number : \c"
          continue
      else
          if [ ${l_sel} -gt 0 -a ${l_sel} -le ${l_sub} ]
          then
               break
          else
               echo "Select number between 1 and ${l_sub} : \c"
          fi
      fi
     done
 fi
}

flash_info()
{
 l_f_nm=$1
 if [ ! -z "${l_f_nm}" ]
 then
   printf "-------------------------------------------------------------------\n"
   printf "Deatil on flash image ${l_f_nm}\n"
  /usr/bin/head -100 ${p_flash_loc}/${l_f_nm} | eval `/usr/bin/nawk '
  {
    if($1 == "section_end=identification") {
        l_line_cnt = NR;
     }
  }
  END { 
    printf("l_line_cnt=%d\n",l_line_cnt); 
  }'`
  /usr/bin/head -${l_line_cnt} ${p_flash_loc}/${l_f_nm}
   printf "-------------------------------------------------------------------\n"
fi
}

record_flash()
{
 l_f_nm=$1
 /usr/bin/sed -e '/p_flash_file=/d' ${p_d_env_file} > ${p_d_env_file}.tmp
 [ $? -ne 0 ] && Print_MSG WARNING "Failed recording the flash file"
/usr/bin/mv ${p_d_env_file}.tmp ${p_d_env_file}
 echo "p_flash_file=\"${l_f_nm}\";export p_flash_file" >> ${p_d_env_file}
 [ $? -ne 0 ] && Print_MSG WARNING "Failed recording the flash file"
}

accept_flash()
{
  echo "\n=========================================================================================="
  echo "List of available archives for ${p_arch} at ${p_flash_loc}\n"
  
  l_chk=`/bin/ls -1 ${p_flash_loc}/s*_*_${p_proc_type}_${p_arch}_*.flar 2>&1 >/dev/null`
  if test -n "${l_chk}"
  then
      Print_MSG ERROR "s*_*_${p_proc_type}_${p_arch}_*.flar ${l_chk}"
      exit 1
  fi

  let l_sub=0 
  set -A l_flash
  for l_f_nm in $( cd ${p_flash_loc}; /bin/ls -1 s*_*_${p_proc_type}_${p_arch}_*.flar | /usr/bin/nawk -F\_ ' { print $5 } ' | /usr/bin/nawk -F\. ' { print $1 }' | sort -Mr)
  do
    l_f_nm=`/bin/ls -1 ${p_flash_loc}/s*_*_${p_proc_type}_${p_arch}_${l_f_nm}.flar`
    l_f_nm=`/usr/bin/basename ${l_f_nm}`
    l_sub=`expr ${l_sub} + 1`
    l_flash[$l_sub]=${l_f_nm}
  done

  printf "no%3shw%4sos%3sos-version%9screated%2sfile-name%12s\n"
  printf "----%1s-----%1s----%1s------------------%1s--------%1s----------------------------------\n" 

  
  select_rows

  let l_a=1
  let l_b=1
 
  while [ ${l_a} -le ${l_sub} ]
  do
    if [ `expr ${l_a} + ${l_rows}` -ge ${l_sub} ]
    then
       while [ ${l_a} -le ${l_sub} ] 
       do
          desc_flash ${l_flash[$l_a]} ${l_a}
          l_a=`expr ${l_a} + 1`
       done
          l_choice="S"
          select_flash
    else
       l_b=`expr ${l_a} + ${l_rows}`
       while [ ${l_a} -le ${l_b} ]
       do
          desc_flash ${l_flash[$l_a]} ${l_a}
          l_a=`expr ${l_a} + 1`
       done
          disp_flash
          select_flash
    fi
  done

  l_flash_file=${l_flash[$l_sel]}

  if [ ! -z "${l_flash_file}" ]
  then
      while true
      do
        echo "You have selected flash image ${l_flash_file}\n"
        flash_info ${l_flash_file}
        echo "Enter yes to continue or no to select other flash image : \c"
        read l_ans
        Print_RSP ${l_ans}
        l_ans=`echo ${l_ans}|tr "[a-z]" "[A-Z]"`
        if [[ "${l_ans}" = "Y" || "${l_ans}" = "YES" ]]
        then
            record_flash ${l_flash_file}
            break
        else
            if [[ "${l_ans}" = "N" || "${l_ans}" = "NO" ]]
            then
                select_flash
                l_flash_file=${l_flash[$l_sel]}
            fi
        fi
      done
  fi
}

update_status()
{
 l_var=$1
 /usr/bin/sed -e '/p_mig_flag=/d' ${p_d_env_file} > ${p_d_env_file}.tmp
 [ $? -ne 0 ] && Print_MSG WARNING "Failed updating the completion status"
/usr/bin/mv ${p_d_env_file}.tmp ${p_d_env_file}
 echo "p_mig_flag=\"${l_var}\";export p_mig_flag" >> ${p_d_env_file}
 [ $? -ne 0 ] && Print_MSG WARNING "Failed updating the completion status"
}

check_duplicate()
{
  l_this_pid=$$
  l_this_tty=`/usr/bin/ps -o tty -p $$ | /usr/bin/nawk '{ if($1!="TT") print $1; }'`
  l_upg_running=`/usr/bin/ps -ef | /usr/bin/grep -wv grep | /usr/bin/grep "ksh .*${l_script_name}.sh" \
     | /usr/bin/nawk -v "l_this_pid=${l_this_pid}" -v "l_this_tty=${l_this_tty}" '
        BEGIN { l_pcnt = 0; }
        { 
          if($2==l_this_pid || $3==l_this_pid || $6 == l_this_tty) next;
          else l_pcnt++;
        }
        END { print l_pcnt; }'`
  if [ ${l_upg_running} -gt 0 ]
  then
      #Check again to reduce probability of wrong answer using ps command
    sleep 2
    l_upg_running=`/usr/bin/ps -ef | /usr/bin/grep -wv grep | /usr/bin/grep "ksh .*${l_script_name}.sh" | /usr/bin/grep -wcv " ${l_this_pid}"`
      if [ ${l_upg_running} -gt 0 ]
      then
          Print_MSG INFO "${l_script_name} is already running"
          exit 0
      fi
  fi
}

new_fs()
{
 l_p=$1
        printf "\n     newfs -v ${l_p}\n\n"
/usr/sbin/newfs -v ${l_p} <<:END:
y
:END:
 check_error $? "Running newfs on ${l_p}"
}

write_done()
{
 l_s=$1
 /usr/bin/touch ${l_done_dir}/${l_s}
}

run_lustatus()
{
 l_be=$1
 l_be_status=0
 if [ -f "/usr/sbin/lustatus" ]
 then
    /usr/sbin/lustatus ${l_be} > /dev/null 2>&1
    l_be_status=`echo $?`
 else
    Print_MSG ERROR "lustatus not found"
    exit 1
 fi
}

run_lucreate()
{
  if [ -f "/usr/sbin/lucreate" ]
  then
      run_lustatus ${l_curr_be}
        if [ ${l_be_status} -eq 0 ]
        then

              printf "     lucreate -s - -m /:${l_alt_be_disk}${l_root_p}:ufs 
                                         -m -:${l_alt_be_disk}${l_swap_p}:swap
                                         -m /var:${l_alt_be_disk}${l_var_p}:ufs
                                         -m /home:${l_alt_be_disk}${l_home_p}:ufs
                                         -n ${p_alt_be_name} \n\n"

              /usr/sbin/lucreate -s - -m /:${l_alt_be_disk}${l_root_p}:ufs  \
                                 -m -:${l_alt_be_disk}${l_swap_p}:swap      \
                                 -m /var:${l_alt_be_disk}${l_var_p}:ufs     \
                                 -m /home:${l_alt_be_disk}${l_home_p}:ufs   \
                                 -n ${p_alt_be_name} 
              check_error $? ${l_step_name}
        else
            if [ ${l_be_status} -eq 1 ]
            then

              printf "     lucreate -c ${l_curr_be} -s - -m /:${l_alt_be_disk}${l_root_p}:ufs
                                          -m -:${l_alt_be_disk}${l_swap_p}:swap
                                          -m /var:${l_alt_be_disk}${l_var_p}:ufs
                                          -m /home:${l_alt_be_disk}${l_home_p}:ufs
                                          -n ${p_alt_be_name} \n\n"

              /usr/sbin/lucreate -c ${l_curr_be} -s - -m /:${l_alt_be_disk}${l_root_p}:ufs  \
                                 -m -:${l_alt_be_disk}${l_swap_p}:swap      \
                                 -m /var:${l_alt_be_disk}${l_var_p}:ufs     \
                                 -m /home:${l_alt_be_disk}${l_home_p}:ufs   \
                                 -n ${p_alt_be_name} 
              check_error $? ${l_step_name}
            fi
        fi
  else
      Print_MSG ERROR "lucreate not found"
      exit  1
  fi
      
}

run_luupgrade()
{

 if [[ -z "${l_flash_file}" && ! -z "${p_flash_file}" ]]
 then
     l_flash_file=${p_flash_file}
 fi
 if [ -f "/usr/sbin/luupgrade" ]
 then

    printf "     luupgrade -f -n ${p_alt_be_name} -s ${p_ossw_loc} 
                              -a ${p_flash_loc}/${l_flash_file} \n\n"

    /usr/sbin/luupgrade -f -n ${p_alt_be_name} -s ${p_ossw_loc} \
                        -a ${p_flash_loc}/${l_flash_file} 
    check_error $? ${l_step_name}
 else
     Print_MSG ERROR "luupgrade not found"
     exit  1
 fi
}

run_lumount()
{
 l_be=$1
 if [ -f "/usr/sbin/lumount" ]
 then
    /usr/sbin/lumount ${l_be} /.alt.${l_be} > /dev/null
    check_error $? "could not mount ${l_be}"
 else
    Print_MSG ERROR "lumount not found"
    exit  1
 fi
}

create_smmsp_user()
{
 l_m="smmsp"
 l_u_e=`/usr/bin/grep -c ${l_m} /etc/passwd`
 if [ ${l_u_e} -eq 0 ]
 then
     /usr/sbin/groupadd  -g 25 -o ${l_m} > /dev/null 2>&1
     if [ $? -eq 0 ]
     then
        /usr/sbin/useradd -u 25 -o -g 25 -s "/usr/bin/true" -d "/" -c "SendMail Message Submission Program" ${l_m} > /dev/null 2>&1
        if [ $? -ne 0 ]
        then
            Print_MSG WARNING "PLEASE FOLLOW PROCESS GUIDE TO CREATE USER ${l_m}"
        fi
     else
            Print_MSG WARNING "PLEASE FOLLOW PROCESS GUIDE TO CREATE USER ${l_m}"
     fi
 fi
}

process_system()
{
l_file_name=$1
/usr/bin/cat ${l_obs_param_file} | /usr/bin/egrep -v '^#|^$' | /usr/bin/nawk -F'|' ' {printf "s/%s/%s/\n",$1,$2}' \
> ${l_obs_param_file}.tmp
 if [ -s "${l_obs_param_file}.tmp" ]
 then
     /usr/bin/sed -f ${l_obs_param_file}.tmp ${l_file_name} > ${l_file_name}.tmp
     check_error $? "Failed processing ${l_file_name}"
     /usr/bin/cmp -s ${l_file_name} ${l_file_name}.tmp
     if [ $? -eq 1 ]
     then
         /usr/bin/cp ${l_file_name}.tmp ${l_file_name}
         check_error $? "Failed copying ${l_file_name}"
     fi
 fi
 /usr/bin/rm -f ${l_file_name}.tmp
}


process_vfstab()
{
l_rd=$1
l_rrd=$2
l_vd=$3
l_vrd=$4
l_sd=$5

l_file_name=$6

/usr/bin/nawk -v "ROOT_DSK=${l_rd}" -v "ROOT_RDSK=${l_rrd}" \
          -v "VAR_DSK=${l_vd}" -v "VAR_RDSK=${l_vrd}"   \
          -v "SWAP_DSK=${l_sd}" '
BEGIN {
    swap_done = 0;
    root_done = 0;
    var_done = 0;
    usr_done = 0;
    opt_done = 0;

    if(ROOT_DSK=="" || ROOT_RDSK=="") {
         exit 1;
    }
    if(VAR_DSK=="" || VAR_RDSK=="") {
         exit 1;
    }
    if(SWAP_DSK=="") {
         exit 1;
    }
}

{
    if(/^[ 	]*#/ || /^$/ ) {
        printf("%s\n",$0);
        next;
    }
    else if($3 == "/" && root_done == 0) {
        printf("##%s\n",$0);
        printf("%s\t%s\t%s\tufs\t1\tno\t-\n",ROOT_DSK,ROOT_RDSK,"/");
        root_done = 1;
        next;
    }
    else if($3 == "/var" && var_done == 0) {
        printf("##%s\n",$0);
        printf("%s\t%s\t%s\tufs\t1\tno\t-\n",VAR_DSK,VAR_RDSK,"/var");
        var_done = 1;
        next;
    }
    else if($3 == "/usr" && usr_done == 0) {
        printf("##%s\n",$0);
        usr_done = 1;
        next;
    }
    else if($3 == "/opt" && opt_done == 0) {
        printf("##%s\n",$0);
        opt_done = 1;
        next;
    }
    else if($4 == "swap" && swap_done == 0) {
        printf("##%s\n",$0);
        printf("%s\t%s\t%s\tswap\t-\tno\t-\n",SWAP_DSK,"-","-");
        swap_done = 1;
        next;
    }
    else {
        printf("%s\n",$0);
    }
} 
END {
    if (var_done == 0) {
        printf("%s\t%s\t%s\tufs\t1\tno\t-\n",VAR_DSK,VAR_RDSK,"/var");
        var_done = 1;
    }
 }' ${l_file_name}

}

find_param()
{
 l_p=$1
 l_f=$2
   /usr/bin/grep ${1} ${2} > /dev/null 2>&1
   l_rc=$?
}

derive_system_locale()
{
 l_f=$1
 l_p=$2
 if [ -z "${LANG}" ]
 then
     l_lc=`/usr/bin/grep ^LANG "/etc/default/init" | /usr/bin/awk -F\= ' {print $2}'`
     if [ -z "${l_lc}" ]
     then
         echo "${l_p}=C" >> ${l_f}
     else
          if [ -d "/.alt.${p_alt_be_name}/usr/lib/locale/${l_lc}" ]
          then
               echo "${l_p}=${l_lc}" >> ${l_f}
          else
               echo "${l_p}=C"       >> ${l_f}
          fi
     fi
 else
     echo "${l_p}=${LANG}" >> ${l_f}
 fi
}

derive_timezone()
{
 l_f=$1
 l_p=$2
 if [ -z "${TZ}" ]
 then
    if [ -f "/etc/default/init" ]
    then
        l_tz=`/usr/bin/grep ^TZ "/etc/default/init" | awk -F\= ' {print $2}'`
        if [ -z "${l_tz}" ]
        then
            echo "${l_p}=US/Pacific" >> ${l_f}
        else
            echo "${l_p}=${l_tz}"    >> ${l_f}
        fi
    else
        echo "${l_p}=US/Pacific" >> ${l_f}
    fi
 else
     echo "${l_p}=${TZ}" >> ${l_f}
 fi
}

derive_terminal()
{
 l_f=$1
 l_p=$2
 if [ -f "/etc/.sysIDtool.state" ]
 then
     l_t=`tail -1 /etc/.sysIDtool.state`
     if [ -z "${l_t}" ]
     then
         echo "${l_p}=sun-cmd" >> ${l_f}
     else
         l_a=`echo ${l_t}|cut -c1`
         if [ -f "/usr/share/lib/terminfo/${l_a}/${l_t}" ]
         then
             echo "${l_p}=${l_t}"  >> ${l_f}
         else
             echo "${l_p}=sun-cmd" >> ${l_f}
         fi
     fi
  else
      echo "${l_p}=sun-cmd" >> ${l_f}
  fi
}

derive_timeserver()
{
 l_f=$1
 l_p=$2
 echo "${l_p}=localhost" >> ${l_f}
}

derive_security_policy()
{
 l_f=$1
 l_p=$2
 echo "${l_p}=NONE" >> ${l_f}
}

process_sysidcfg()
{
 l_file_name=$1
 if [ -f "${l_file_name}" ]
 then
     for l_param in `/usr/bin/cat ${l_sysidcfg_file} | /usr/bin/egrep -v '^#|^$' `
     do
       find_param ${l_param} ${l_file_name}
       if [ ${l_rc} -eq 1 ]
       then
           derive_${l_param} ${l_file_name} ${l_param}
       fi
     done
 else
     for l_param in `/usr/bin/cat ${l_sysidcfg_file} | /usr/bin/egrep -v '^#|^$' `
     do
       derive_${l_param} ${l_file_name} ${l_param}
     done
 fi
}

process_statedb()
{
   l_f_nm=$1
   l_state_db=`/usr/bin/egrep -i '^set md:mddb_bootlist' ${l_f_nm}| \
       /usr/bin/awk -F\= '{ print $2 }'|/usr/bin/sed -e 's/"//g'`
   l_state_db=${l_state_db:-0}
   if [ ! -z "${l_state_db}" ]
   then
         print "# Begin MDD database info (do not edit)"
         print "mddb_bootlist1=\"\c"
         for l_a in `/usr/bin/egrep -i '^set md:mddb_bootlist' ${l_f_nm}|/usr/bin/awk -F\= '{ print $2 }'|/usr/bin/sed -e 's/"//g'`
         do
            print "$l_a:id0 \c"
         done
         print "\";"
         print "md_devid_destroy=1;"
         print "# End MDD database info (do not edit)"
   fi
}

process_sds()
{
 l_f_nm=$1
 [ -f /.alt.${p_alt_be_name}/etc/lvm/md.cf ] && \
 /usr/bin/cp -p /.alt.${p_alt_be_name}/etc/lvm/md.cf /.alt.${p_alt_be_name}/etc/lvm/md.cf.${p_alt_be_name}
 [ $? -ne 0 ] && Print_MSG WARNING "Failed taking backup of md.cf in ABE" 
 [ -f /.alt.${p_alt_be_name}/etc/lvm/md.tab ] && \
 /usr/bin/cp -p /.alt.${p_alt_be_name}/etc/lvm/md.tab /.alt.${p_alt_be_name}/etc/lvm/md.tab.${p_alt_be_name}
 [ $? -ne 0 ] && Print_MSG WARNING "Failed taking backup of md.tab in ABE"
 [ -f /.alt.${p_alt_be_name}/kernel/drv/md.conf ] && \
 /usr/bin/cp -p /.alt.${p_alt_be_name}/kernel/drv/md.conf /.alt.${p_alt_be_name}/kernel/drv/md.conf.${p_alt_be_name}
 [ $? -ne 0 ] && Print_MSG WARNING "Failed taking backup of md.conf in ABE"

 if [ `/usr/bin/which metastat|/usr/bin/grep -c SUNWmd` -eq 1\
      -a `/usr/bin/which metastat|/usr/bin/grep -c 'no metastat'` -eq 0 ]
 then
    printf "     Working on sds...\n"
    /usr/bin/cp -p /etc/opt/SUNWmd/md.tab /.alt.${p_alt_be_name}/etc/lvm/md.tab
    check_error $? "Failed to copy md.tab"
    l_sds_exist="Y"
 else
     if [ -x `/usr/bin/which metastat` \
          -a `/usr/bin/which metastat|/usr/bin/grep -c 'no metastat'` -eq 0 ]
     then  
         printf "     Working on sds...\n"
         /usr/bin/cp -p /etc/lvm/md.tab /.alt.${p_alt_be_name}/etc/lvm/md.tab
	 check_error $? "Failed to copy md.tab"
         l_sds_exist="Y"
     else
         :
     fi
 fi 
 l_sds_exist=${l_sds_exist:-"N"}
     
 if [[ -f "${l_f_nm}" && "${l_sds_exist}" = "Y" ]]
 then
     l_md_cf="/.alt.${p_alt_be_name}/etc/lvm/md.cf"
     if [ `metastat -p|/usr/bin/wc -l` -ne 0 ]
     then
         print "# metadevice configuration file" > ${l_md_cf}
         print "# do not hand edit"             >> ${l_md_cf}
         metastat -p                            >> ${l_md_cf}
         check_error $? "Failed to create md.cf in ABE"
     fi
     l_md_conf="/.alt.${p_alt_be_name}/kernel/drv/md.conf"
     process_statedb ${l_f_nm} >> ${l_md_conf}
 fi
}

process_lofs()
{
l_cycle=1
for l_lofs in `/usr/bin/nawk '{ if ($3 == "lofs") print $2; }' /etc/mnttab`
do
if [ ${l_cycle} -eq 1 ]
then
   printf "     Working on lofs mount points...\n"
   l_cycle=`expr ${l_cycle} + 1`
fi
echo $l_lofs | eval `/usr/bin/nawk ' 
    {
      x=split($0,p,"/"); 
      for ( i = 1;i < x;++i )
        l_d = l_d p[i] "/"; 
        printf("l_d=%s\n",l_d);
    }'`
l_m=`/usr/sbin/df ${l_d} | /usr/bin/awk ' { print $1 } '`
if [[ "${l_m}" = "/" || "${l_m}" = "/var" ]]
then
    /usr/bin/mkdir -p /.alt.${p_alt_be_name}${l_lofs}
    check_error $? "Failed Creating ${l_lofs}"
fi
done
}

set_misc_env()
{
  l_alt_be_disk="/dev/dsk/${p_alt_be_disk}"
  l_alt_be_rdisk="/dev/rdsk/${p_alt_be_disk}"
  l_root_p="s0"
  l_swap_p="s1"
  l_var_p="s5"
  l_home_p="s6"
  l_alt_dsk_dir="/.alt.${p_alt_be_name}/dev/dsk"
  l_alt_rdsk_dir="/.alt.${p_alt_be_name}/dev/rdsk"
  l_alt_vxvm_dir="/.alt.${p_alt_be_name}/var/vxvm"
  l_alt_lic_dir="/.alt.${p_alt_be_name}/etc/vx"
  l_alt_system_file="/.alt.${p_alt_be_name}/etc/system"
  l_alt_vfstab_file="/.alt.${p_alt_be_name}/etc/vfstab"
  l_alt_dfstab_file="/.alt.${p_alt_be_name}/etc/dfs/dfstab"
  l_alt_instdb_file="/.alt.${p_alt_be_name}/etc/vx/reconfig.d/state.d/install-db"
  l_alt_volboot_file="/.alt.${p_alt_be_name}/etc/vx/volboot"
  l_alt_sysidcfg_file="/.alt.${p_alt_be_name}/etc/sysidcfg"
  l_lic_dir="/etc/vx"
  l_vxvm_dir="/var/vxvm"
  l_system_file="/etc/system"
  l_vfstab_file="/etc/vfstab"
  l_dfstab_file="/etc/dfs/dfstab"
  l_volboot_file="/etc/vx/volboot"
}

set_bit_mode()
{
  l_mode=$1
  if [ -x "/usr/sbin/eeprom" ]
  then
      if [ "${l_mode}" = "32-bit" ]
      then
         /usr/sbin/eeprom boot-file="kernel/unix"
         check_error $? "Not able to set $l_mode} mode"
         Print_MSG INFO "CBE is ${l_mode} mode ABE is set to ${l_mode} mode"
      else
         Print_MSG INFO "CBE is ${l_mode} mode ABE is set to ${l_mode} mode"
      fi
   else
         Print_MSG WARNING "eeprom COMMAND NOT FOUND"
         Print_MSG WARNING "SYSTEM MAY BOOT 64-bit KERNEL"
  fi
}

initliaze_files()
{
 l_init_list="/var/adm/utmpx \
              /var/adm/wtmpx"

 for file in `echo ${l_init_list}`
 do
   > /.alt.${p_alt_be_name}${file} > /dev/null 2>&1
 done
}

remove_files()
{
 l_rm_list="/var/mail/root \
            /var/adm/messages \
            /var/adm/lastlog \
            /var/vx/isis/vxisis.lock \
            /etc/mail/aliases.db \
            /etc/mail/aliases.dir \
            /etc/mail/aliases.pag"

for file in `echo ${l_rm_list}`
do 
  /usr/bin/rm -f /.alt.${p_alt_be_name}${file} > /dev/null 2>&1
done
}

process_files()
{
 l_dir=`/usr/bin/basename $1`
 l_file=$2
 l_cbe_file="${1}/${l_file}"
 l_abe_file="/.alt.${p_alt_be_name}${1}/${l_file}"
 l_exclude_file="${l_exclude_dir}/${l_dir}_${l_file}.exclude"

 if [ -f "${l_exclude_file}" ]
 then
     if [ -f "${l_abe_file}" ]
     then
         /usr/bin/cp -p ${l_abe_file} ${l_abe_file}.${p_alt_be_name}
         check_error $? "Failed backing up ${l_abe_file}"
         /usr/bin/cp -p ${l_cbe_file} ${l_abe_file}.${l_curr_be}
         check_error $? "Failed backing up ${l_cbe_file}"
         /usr/bin/egrep -v -f ${l_exclude_file} ${l_cbe_file} >> ${l_abe_file}
         [ $? -gt 1 ] && Print_MSG WARNING "Failed processing ${l_cbe_file}"
     fi
 fi
}

process_list()
{
 l_process_list="/etc|aliases \
                 /etc|services \
                 /etc|inetd.conf"

 for file_name in `echo ${l_process_list}`
 do
   l_dir=`echo ${file_name} |/usr/bin/awk -F\| '{ print $1}'`
   l_file=`echo ${file_name}|/usr/bin/awk -F\| '{ print $2}'`
   process_files ${l_dir} ${l_file}
 done
}

warn_message()
{
  printf "  WARNING WARNING \n\n"
  printf "  DUMPDEVICE IS CONFIGURED ON ${p_alt_be_disk} \n"
  printf "  luconfig MAY FAIL INCASE OF FAILURE PLEASE   \n"
  printf "  RECONFIGURE DUMPDEVICE USING dumpadm COMMAND \n"
  printf "  THEN CONTINUE WITH s9_migration STEP \n\n"
  printf "  <Hit Enter to Continue : \c"
  read l_response
  Print_RSP ${l_response}
}

chk_dump_device()
{
 [ -f "/etc/dumpadm.conf" ] && [ `/usr/bin/grep -c "${p_alt_be_disk}" /etc/dumpadm.conf` -gt 0 ] \
 && warn_message
}

chk_raw_devices()
{
  l_err=0
  for file in `ls -1 /dev/rdsk/*`
  do
    l_owner=`ls -lL ${file} | /usr/bin/nawk ' { print $3 } '`
    l_group=`ls -lL ${file} | /usr/bin/nawk ' { print $4 } '`
    /usr/bin/chown ${l_owner}:${l_group} /.alt.${p_alt_be_name}${file}
    [ $? -ne 0 ] && l_err=1
  done
  [ ${l_err} -ne 0 ] && Print_MSG ERROR "Failed changing owner on raw devices"
   
  l_err=0
  if test -c "/dev/md/rdsk/*"
  then
    for file in `ls -1 /dev/md/rdsk/*`
    do
      l_owner=`ls -lL ${file} | /usr/bin/nawk ' { print $3 } '`
      l_group=`ls -lL ${file} | /usr/bin/nawk ' { print $4 } '`
      /usr/bin/chown ${l_owner}:${l_group} /.alt.${p_alt_be_name}${file}
      [ $? -ne 0 ] && l_err=1
    done
    [ ${l_err} -ne 0 ] && Print_MSG ERROR "Failed changing owner on raw md devices"
  fi
}

terminate_program()
{
    echo "\n-----------------------------------------------------------------------"
    echo "[`${DATE}`] WARNING : Terminating on user request"
    l_pid=`/usr/bin/ps -ef | /usr/bin/grep -v grep | /usr/bin/grep ${l_script_name} | /usr/bin/awk ' { print $2 } '`
    /usr/bin/kill ${l_pid} > /dev/null 2>&1
    echo "[`${DATE}`] WARNING : Terminated ${l_script_name}"
    echo "** Please check for any child processes"
    exit 2
}

# Done defining all modules

trap terminate_program 2 3

read_d_env
set_misc_env


if [ ! -d ${p_ossw_loc} ]
then
    Print_MSG ERROR "${p_ossw_loc} does not exist"
    exit 1
fi

if [ "${p_premig_flag}" = "Y" ]  #{
then

    if [ "${p_mig_flag}" = "Y" ]
    then 
        Print_MSG INFO "s9_migration has already ran...."
        exit 0
    fi

# Before starting make sure it is not already running

check_duplicate

#Step - 1 Verify Disk Slices.
 
 l_oslevel=`eval echo ${p_oslevel} | awk -F\. ' { print $2 } '`

 if [ ${l_oslevel} -le "5" ]
 then
      Print_MSG ERROR "${p_oslevel} is not supported"
      exit 1
 else
      l_curr_be="S${l_oslevel}BE"
 fi

l_step_name="RUNS9_DISK_SLICES"
check_done ${l_step_name}

Print_MSG START ${l_script_name}

if [ ${l_done_flag} = "N" ]
then

    accept_disk
    accept_flash
    set_misc_env

    Print_MSG BEGIN ${l_step_name} 

    printf "  Running make file system ...\n\n"

        #run newfs for non swap partitions
        new_fs ${l_alt_be_disk}${l_root_p}
        new_fs ${l_alt_be_disk}${l_var_p}
        new_fs ${l_alt_be_disk}${l_home_p}

       write_done ${l_step_name}

    Print_MSG END ${l_step_name}
fi

#Step - 2 Run lucreate
  
      check_done ${l_step_name}

      l_step_name="RUNS9_LUCREATE"

      check_done ${l_step_name}
  
 if [ ${l_done_flag} = "N" ]
 then 
   Print_MSG BEGIN ${l_step_name}

       chk_dump_device

   printf "  Creating boot environment ...\n\n"

       run_lucreate

   Print_MSG END ${l_step_name}
 fi
       write_done ${l_step_name}

#Step - 3 Run luupgrade
     
      check_done ${l_step_name}

      l_step_name="RUNS9_LUUPGRADE"

      check_done ${l_step_name}

 if [ ${l_done_flag} = "N" ]
 then
   Print_MSG BEGIN ${l_step_name}

      create_smmsp_user

   printf "  Installing flash image ...\n\n"

      run_luupgrade

   Print_MSG END ${l_step_name}
 fi
      write_done ${l_step_name}
        
#Step - 4 cleaning the known directories

      check_done ${l_step_name}

      l_step_name="RUNS9_COPY_FILES"

      check_done ${l_step_name}

 if [ ${l_done_flag} = "N" ]
 then
   Print_MSG BEGIN ${l_step_name}

   printf "  Running post cleanup ...\n\n"

    run_lumount ${p_alt_be_name}

    set_misc_env

    # Cleaning dsk and rdsk directories in ABE

     if [ -d ${l_alt_dsk_dir} -a -d ${l_alt_rdsk_dir} ]
     then

        printf "     Working on device tree...\n"

        /usr/bin/rm -rf ${l_alt_dsk_dir} ${l_alt_rdsk_dir}
        check_error $? "Failed to remove ${l_alt_dsk_dir} ${l_alt_rdsk_dir}"

        cd /dev;/bin/tar cf - ./rdsk ./dsk | (cd /.alt.${p_alt_be_name}/dev;tar xfp -)
        check_error $? "Failed to copy dev and rdsk directories"

        chk_raw_devices
     fi

    # Copying veritas rootdg file in ABE

    if [ -d "${l_vxvm_dir}" -a -d "${l_alt_vxvm_dir}" ]
    then

        if [ ! -d "${l_alt_vxvm_dir}/tempdb" ]
        then
           /usr/bin/mkdir ${l_alt_vxvm_dir}/tempdb
           check_error $? "Failed to create ${l_alt_vxvm_dir}/temdb"
        fi

        /usr/bin/cp -p ${l_vxvm_dir}/tempdb/rootdg ${l_alt_vxvm_dir}/tempdb/rootdg
        check_error $? "Failed to copy rootdg file"
    fi

    # Copying /etc/system file in ABE

    if [ -f "${l_system_file}" ] 
    then
       
        printf "     Working on system file...\n"

        if [[ -f "${l_alt_system_file}" && ! -f "${l_alt_system_file}.${p_alt_be_name}" ]]
        then
          /usr/bin/cp -p ${l_alt_system_file} ${l_alt_system_file}.${p_alt_be_name} 
          check_error $? "Failed to backup ABE system file" 
        fi

        /usr/bin/cp -p ${l_system_file} ${l_alt_system_file}
        check_error $? "Failed to copy system file"

    # Process the system for obsolete parameters and veritas
        
        process_system ${l_alt_system_file}
      
    fi

    # Copying /etc/dfs/dfstab file in ABE

    if [ -f  "${l_dfstab_file}" ]
    then

        if [[ -f "${l_alt_dfstab_file}" && ! -f "${l_alt_dfstab_file}.${p_alt_be_name}" ]]
        then
          /usr/bin/cp -p ${l_alt_dfstab_file} ${l_alt_dfstab_file}.${p_alt_be_name}
          check_error $? "Failed to backup ABE dfstab file"
        fi
       
        /usr/bin/cp -p ${l_dfstab_file} ${l_alt_dfstab_file}
        check_error $? "Failed to copy dfstab file"
        
    fi

    # Copying /etc/auto_ files in ABE

        printf "     Working on auto files...\n"

    for l_auto_file in `/bin/ls -d1 /etc/auto_*`
    do

       if [[ -f "/.alt.${p_alt_be_name}/${l_auto_file}" && ! -f "/.alt.${p_alt_be_name}/${l_auto_file}.${p_alt_be_name}" ]]
       then
           /usr/bin/cp -p /.alt.${p_alt_be_name}/${l_auto_file} /.alt.${p_alt_be_name}/${l_auto_file}.${p_alt_be_name}
           check_error $? "Failed to backup ABE ${l_auto_file}"
       fi
       if [ -f "${l_auto_file}" ]
       then 
           /usr/bin/cp -p ${l_auto_file} /.alt.${p_alt_be_name}/${l_auto_file}
           check_error $? "Failed to copy ${l_auto_file}"
       fi
    done

    # Copying /.rhosts file in ABE

    if [ -f "/.rhosts" ]
    then
         if [[ -f "/.alt.${p_alt_be_name}/.rhosts" && ! -f "/.alt.${p_alt_be_name}/.rhosts.${p_alt_be_name}" ]]
         then
             /usr/bin/cp -p  /.alt.${p_alt_be_name}/.rhosts /.alt.${p_alt_be_name}/.rhosts.${p_alt_be_name}
             check_error $? "Failed to backup ABE .rhosts"
         fi

     /usr/bin/cp -p /.rhosts /.alt.${p_alt_be_name}/.rhosts
     check_error $? "Failed to copy .rhosts"

    fi

    # Copy the veritas licences files in ABE

    if [ -d "${l_lic_dir}/elm" -o -d "${l_lic_dir}/licenses/lic" ]
    then

        printf "     Working on veritas licenses...\n"
     
        if [ ! -d "${l_alt_lic_dir}/elm" ]
        then
            /usr/bin/mkdir ${l_alt_lic_dir}/elm
            check_error $? "Failed to create ${l_alt_lic_dir}/elm"
        fi
        if [ ! -d "${l_alt_lic_dir}/licenses/lic" ]
        then
            /usr/bin/mkdir ${l_alt_lic_dir}/licenses;/usr/bin/mkdir ${l_alt_lic_dir}/licenses/lic
            check_error $? "Failed to create ${l_alt_lic_dir}/licenses"
        fi

        if [ -d "${l_lic_dir}/elm" ]
        then
            /usr/bin/cp -p ${l_lic_dir}/elm/* ${l_alt_lic_dir}/elm
        fi
        if [ -d "${l_lic_dir}/licenses/lic" ]
        then
            /usr/bin/cp -p ${l_lic_dir}/licenses/lic/* ${l_alt_lic_dir}/licenses/lic
        fi
    fi

    # Dropping install-db file

    if [ -f "${l_alt_instdb_file}" ]
    then
        /usr/bin/rm ${l_alt_instdb_file}
        check_error $? "Failed removing ${l_alt_instdb_file}"
    fi

    # Copying volboot file

    if [ -f "${l_volboot_file}" -a ! -f "${l_alt_volboot_file}" ]
    then 
        /usr/bin/cp -p ${l_volboot_file} ${l_alt_volboot_file}
        check_error $? "Failed copying ${l_alt_volboot_file}"
    fi
        
    # Process /etc/vfstab file in ABE
  
    # Backup the original file and copy over the file from CBE

    if [[ -f "${l_alt_vfstab_file}" && ! -f "${l_alt_vfstab_file}.${p_alt_be_name}" ]]
    then
        /usr/bin/cp -p ${l_alt_vfstab_file} ${l_alt_vfstab_file}.${p_alt_be_name}
        check_error $? "Fail to backup ${l_alt_vfstab_file}"
    fi
        /usr/bin/cp -p ${l_vfstab_file} ${l_alt_vfstab_file}
        check_error $? "Fail to copy ${l_vfstab_file}"

    # Process the file
        printf "     Working on vfstab...\n"

    process_vfstab "${l_alt_be_disk}${l_root_p}" "${l_alt_be_rdisk}${l_root_p}" \
                   "${l_alt_be_disk}${l_var_p}"  "${l_alt_be_rdisk}${l_var_p}"  \
                   "${l_alt_be_disk}${l_swap_p}" "${l_alt_vfstab_file}" > ${l_alt_vfstab_file}.tmp
    if [ $? -eq 0 ]
    then
        /usr/bin/mv ${l_alt_vfstab_file}.tmp ${l_alt_vfstab_file}
    else
        /usr/bin/cp -p ${l_alt_vfstab_file}.${p_alt_be_name} ${l_alt_vfstab_file}
        Print_MSG ERROR "Failed processing ${l_alt_vfstab_file}"
        exit 1
    fi

    # Process sds stuff
 
    process_sds ${l_system_file} 
    check_error $? "Failed processing md.conf"

    # Process sysidcfg file in ABE
        printf "     Working on sysidcfg...\n"
    process_sysidcfg ${l_alt_sysidcfg_file}
    check_error $? "Failed preconfiguring sysidcfg"
 
    # Process lofs mount points
    process_lofs
    check_error $? "Failed creating lofs mount points"

    # Run Cleanup in ABE
        printf "     Working on miscellaneous files...\n"
    initliaze_files
    remove_files

    # Process files in ABE
        printf "     Working on special files...\n"
    process_list

    # Set bit mode in ABE

        printf "     Working on setting bit mode...\n"
    if [ "${p_oslevel}" = "5.6" ]
    then
        set_bit_mode "32-bit"
    else
        l_curr_mode=`/usr/bin/isainfo -kv 2>/dev/null |awk '{print $1}'`
        l_curr_mode=${l_curr_mode:-"32-bit"}
        if [ "${l_curr_mode}" = "32-bit" ]
        then
            set_bit_mode "${l_curr_mode}"
        else
            set_bit_mode "${l_curr_mode}"
        fi
    fi
        
    run_luumount ${p_alt_be_name}

    Print_MSG END ${l_step_name}
 fi

      write_done ${l_step_name}


 Print_MSG FINISH "Successful ${l_script_name}"
 update_status "Y"
 echo "  Please refer to the S9 Migration Process Guide document"
 echo "  for instructions on proceeding to the next step"
 echo ""

#}
else
    Print_MSG ERROR "PreMigration Step is Not Completed"
fi

exit 0
