#! /usr/dt/bin/dtksh
## 
## Copyright (c) 2000-2001 by Sun Microsystems, Inc.
## All rights reserved.
##
## Permission to use, copy, modify, and distribute this software for any
## purpose with or without fee is hereby granted, provided that the above
## copyright notice and this permission notice appear in all copies.
## 
## THE SOFTWARE IS PROVIDED "AS IS" AND SUN MICROSYSTEMS DISCLAIMS ALL
## WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
## WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL SUN
## MICROSYSTEMS BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
## CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
## OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
## NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
## CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
##
## Netscape is a registered trademark of Netscape Communications
## Corporation in the United States and other countries.
##
## Sun, Sun Microsystems and Solaris are registered trademarks of Sun
## Microsystems, Inc. in the United States and other countries.

## Why dtksh?  Its Version M-12/28/93d while ksh is Version M-11/16/88i
## And I like to use the 93 options.  see ksh93(1)
## @(#) setup_native_LDAP.sh 1.11@(#)
## @(#) iDS 4.x server setup for native LDAP, 01/10/10
## Written by Stacey.Marshall@sun.com and Michael.Haines@sun.com
## Latest version available from http://www.sun.com/blueprints/tools
## Special thanks to:
## Brent Paulson, for putting me back on the tracks when ksh derailed me.
## Authors of the "Native LDAP I" course for the well documented examples.
## #####################################################################

function showInfo {
    case "$1" in
    help) cat <<EOF

DESCRIPTION

Updates Netscape[TM] Directory Server for use with Native LDAP phase I.
Introduced in the Solaris[TM] 8 operating environment, Native LDAP I
implements LDAP as a naming repository.  Refer to the ldap(1) man page
for further details.

The Directory is modified in different stages, or steps.  For further
details on these steps refer to chapter 5 of "Solaris and LDAP Naming
Services; Deploying LDAP in the Enterprise" by Tom Bialaski and
Michael Haines; ISBN 0-13-030678-9.

You may step through and simply observe these stages by executing the
script without any options.

Tasks completed are:

    Updates standard iDS schema:
    o Changes object class configuration file (slapd.oc.conf).
    o Changes password storage scheme to crypt (slapd.conf).
    o Adds Solaris objects to user object class (slapd.user_oc.conf).
    o Adds Solaris attributes to user attributes (slapd.user_at.conf).

    Indexing:
    o Attribute Indexing for Solaris name service data.
    o Virtual List View (VLV) Indexing for Solaris name service data

    Update Directory:
    o Changes ACI on userdn="ldap:///self" for Solaris user conventions.
    o Adds additional top-level classes including nisDomain.
    o Configures proxy-agent and associated ACI
    o Creates Client Profile (ldap_gen_profile(1M)).

    Administration:
    o Optionally configures Directory Service cache parameters.

OPTIONS

    -?
	Displays usage.

    -c DS home
	Pathname of where Netscape Directory Services installed.

    -D Root dn
	As used with ldapmodify(1).  By default this is read from the
	schema.

    -d [integer]
	Debug.  The higher the integer the more information is
	displayed.

    -h
	help: shows this information.

    -l log-file
	By default a log is kept in /var/tmp/native_ldap.log.  This
	option allows you to specify a different file.

    -p profile-Name
	Name used as profile name to ldap_gen_profile(1).

    -q
	Quite. Does not ask questions if a default answer is
	available. 

    -V
	Apply '-v' (for verbose) option to ldapmodify(1) command.

    -v [integer]
	Verbose.  Defaults to 2.  Sets level of detail to display.

    -w password
	Administrators LDAP password.  Will be prompted for if not
	provided.

    -x
	Execute modifications.  Required to make changes.  If this
	option is not specified then no changes are actually made.

EXAMPLES:

    ${0##*/}
    
	Runs through without actually applying any changes.  The only
	affect is that slapd is started if its not already running.

    ${0##*/} -x

	Configures Directory prompting user at each stage and for all
	necessary data.

    ${0##*/} -x -q -w dirpw -v1 -n uk.sun.com

	Configures Directory prompting user for additional data if
	necessary.  Each stage is not explained (-v1) or prompted for
	(-q). The password and nisDomain name are assigned from the
	command line options (-w and -n) respectively.

    ${0##*/} -x -q -w dirpw -v2 -n uk.sun.com \\
	     -a operators -A 0psRus

	Execute changes using "operators" for the proxy-agent name and
	"0psRus" for the proxy-agent password.  The "-A", "-n" and
	"-w" options are necessary if you do not want the script to
	prompt for input.

FILES

    /var/tmp/native_ldap.log

    config/slapd.oc.conf
    config/slapd.conf

SEE ALSO

    ldap(1), ldapmodify(1), ldapsearch(1), ldap_gen_profile(1M)

EOF
    ;;
    "Welcome") cat <<EOF
	
	This script will help you to configure a Netscape[TM]
	Directory Server v4.x with the necessary components to utilise
	Solaris[TM] operating environment Native LDAP phase I.

	The Directory is modified in diffrent stages, or steps. Since
	the publication of "Solaris and LDAP Naming Services;
	Deploying LDAP in the Enterprise" by Tom Bialaski and Michael
	Haines (ISBN 0-13-030678-9), I have changed the sequence.  The
	sequence now reflects the first twelve stages documented in
	chapter 5 of this book. In addition there is an intialisation
	stage, and an additional stage at the end to modify some
	administration tasks like cache sizes.

	It is recommend that you take a backup of the Directory server
	before commencing.

	Note: The Directory server is stopped and started at different
	stages through out this script.  It is recommended that the
	directory server is not in production during this setup.

EOF
    ;;
    1) cat <<EOF

	This step adds attributes to the Directory Server schema using
	LDIF format.

	For Netscape Directory Server this modification will update the 
	slapd.user_at.conf configuration file.  

	For further information refer to rfc2307bis

EOF
    ;;
    2)
    cat <<EOF

	This step updates the Netscape Directory Server file
	slapd.oc.conf.

	Modifying the ipNetwork object class so that the 'cn'
	attribute is no longer required, but is still a member.

	For this purpose the configuration file slapd.conf must be
	updated directly.  And to take affect slapd will be stopped
	before the update and restarted once the change has been made.

EOF
    ;;
    3)
    cat <<EOF

	This step adds new object classes for use with Native LDAP I.

	For Netscape Directory Server this modification will ultimately
	update the slapd.user_oc.conf configuration file.  However,
	this configuration file is not updated until the server
	(slapd) is stopped.

	For further information refer to rfc2307bis

EOF
    ;;
    4) cat <<EOF

	By default the Netscape Directory Server stores passwords in
	the SHA (SHA-1) format.  As Native LDAP I does not support this format
	the server must be changed to store files in crypt(1).

	For this purpose the configuration file slapd.conf must be
	updated directly.  And to take affect slapd will be stopped
	before the update and restarted once the change has been made.

EOF
    ;;
    5) cat <<EOF

	Here the directory naming context(s) is initialised with its
	domainname and additional containers are added to hold the
	various naming information.  The added containers and
	suggested uses are show in the following table:
           ---------------------------------------------------------------- 
	  | Container Information     | Type 				   |
          |---------------------------+------------------------------------|
	  | ou=Ethers                 | bootparams(4), ethers(4) 	   |
	  | ou=Group 		      | group(4) 			   |
          | ou=Hosts                  | hosts(4), ipnodes(4), publickey(4) |
          | ou=Aliases                | aliases(4) 			   |
          | ou=Netgroup               | netgroup(4)			   |
          | ou=Networks               | networks(4), netmasks(4)	   |
          | ou=People                 | passwd(1), shadow(4), user_attr(4),|
          |                           | audit_user(4), publickey for users |
          | ou=protocols              | protocols(4)			   |
          | ou=rpc                    | rpc(4)				   |
          | ou=Services               | services(4)			   |
          | nismapname=auto_*         | auto_*				   |
	   ----------------------------------------------------------------

EOF
    ;;
    5.1) cat <<EOF

	The default auto-mount containers are auto_home, auto_direct
	and auto_master.  However, you may not require these or may
	like to add them when populating the data.  Hence you are
	given the option here to include them now or later.

	If you decide you don't want them later you can remove them
	using ldapdelete(1).

EOF
    ;;
    6) cat <<EOF

	This step changes the access control information (ACI) on
	the directory naming context (suffix).

	The modified ACI prevents a user from changing certain
	details.  For example their User Identity (UID).  While it
	still allows the user to change their password.

EOF
    ;;
    7) cat <<EOF

	This step changes the access control Information (ACI) on the
	Virtual List View (VLV) control.

	The Netscape Directory Service (slapd) by default only permits
	registered users to access the directory through Virtual List
	View (VLV) controls.  The default needs to be changed to allow
	anonymous user access as used by ldapclient(1M).

EOF
    ;;
    8) cat <<EOF
    
	This step adds a proxy-agent in the ou=profile container.

	The proxyagent is used by Native LDAP I when anonymous access
	is not sufficient and a users credentials are not available.
	For example during user login when the pam_unix authentication
	module needs to access the users encrypted password for
	authentication.

EOF
    ;;
    9) cat <<EOF
	
	This step permits the prxoyagent to read the user password.

EOF
    ;;
    10) cat <<EOF

	Creates Client Profile for easy client configuration.

	To simplify Solaris LDAP client configuration one or more
	client profiles may be created on the directory server.  The
	client may specify which client configuration it wishes to use
	with the ldapclient(1M) command.  The client is also
	configured by default to periodicly, as specified by
	SolarisCacheTTL, update its settings from the directory server
	by reading this profile.

	Here we simply create a single profile.  You may add more
	later if you like.

EOF
    ;;
    11) cat <<EOF

	This step creates attribute indexes to improve search performance.

	The following indexes will be added:

	  membernisnetgroup pres,eq,sub       ipNetworkNumber pres,eq   
	  nisnetgrouptriple pres,eq,sub       ipProtocolNumber pres,eq  
	  memberuid pres,eq 		      oncRpcNumber pres,eq      
	  macAddress pres,eq 		      ipServiceProtocol pres,eq 
	  uid pres,eq 			      ipServicePort pres,eq     
	  uidNumber pres,eq 		      nisDomain pres,eq         
	  gidNumber pres,eq 		      nisMapName pres,eq        
	  ipHostNumber pres,eq 		      mail pres,eq

	  Where 'pres' is short for 'present', 'eq' is 'equal' and
	  'sub' is 'substring'.

	Note that building the indexes may take some time if your
	directory already holds data in the indexed containers.

	The Directory Server will be stopped before creating the
	indexes as specified in the Netscape Directory Service
	configuration guide.

EOF
    ;;
    12) cat <<EOF

	This step adds Virtual List View Indexes for containers that
	it is expected will hold a large number of entries.

	In addition to the attribute indexes the VLV indexes are
	created to improve search performance.  Without these VLV
	indexes the directory server and Native LDAP I may appear to
	be unresponsive.

EOF
    ;;
    12.1) cat <<EOF

	Builds VLV indexes defined in STEP 12.

	The Directory Server will be stopped before creating the
	indexes as specified in the Netscape Directory Service
	configuration guide.

EOF
    ;;
    domainname) cat <<EOF

	Please select or enter the domain name to be associated with
	this configuration.

	This domain name will be stored in the nisDomain attribute of
	the nisDomainObject objectclass which will be defined in the
	root DN entry of the DIT representing the desired domain. This
	information is used by the client when initializing the system
	and refreshing the client profile. During the initialization,
	the client searches for an entry on the LDAP server that has
	the nisDomain matching the desired domain. The DN for the
	entry found will be used as the BaseDN for the naming
	information.
EOF
    ;;
    fqdn) cat <<EOF

	Before selecting one of the following options, you should
	confirm that you can look up the host name of your local
	machine. This host name must then contain a fully qualified
	domain name.  Which should probably be the one that you should
	select below.

EOF
    ;;
    cache) cat <<EOF
    
	Modify the Netscape Directory Server cache values.

	The default Netscape Directory Server v4.x cache settings are
	generally not optimised for the Solaris Native LDAP I Naming
	Service. Especially the 'lookthroughlimit' which should be set
	to '-1' (no limit) for use with Native LDAP I.

	The following table shows the default Netscape Directory
	Server v4.x settings and suggested settings for this server.
	The suggested values are calculated from the average freemem.
	As described in the Directory Server v4.x maintenance and
	Operations course notes:

	    Parameter		Default    Calculated
	    
	    lookthroughlimit: 	5000       ${slapdParams[CALClookthroughlimit]}
	    cachesize: 		1000       ${slapdParams[CALCcachesize]}
	    dbcachesize: 	10000000   ${slapdParams[CALCdbcachesize]}

EOF
    ;;
    addprof) cat<<EOF

	The profile shown has not been added to the Directory Server.
	If you are happy with this profile then select "yes" and the
	profile will be added.  Or if you prefer enter "No" and the
	profile will not be added.  You may add your own profile at a
	later time.  See ldap_gen_profile(1M) for further information.

EOF
    ;;
    not23bit) cat <<EOF

	Netscape Directory Server as of v4.x has only been verified on
	Solaris running in 32bit mode.  By using isainfo(1) it has
	been determined that this system is not running in 32bit mode.

	You may continue if you wish.  But no support is provided.
	
	It is recommended that you exit and reboot in 32bit mode at a
	convenient time for all users.

EOF
    ;;
    safemode) cat <<EOF

	In safe mode no updates are made.  This allows you to work
	through and see what questions are asked and what changes
	would be made. However, because some later commands rely on
	previous changes errors may be reported.

	Exit this configuration script and append the '-x' option to
	the command line when you are ready to have the necessary
	commands executed.

EOF
    ;;
    proxyagent) cat <<EOF

	The name given here will be used to create an account in the
	ou=profile container.  Please only enter the name here which
	will be assigned to the 'cn' attribute.

	For example if you use 'proxyagent' as the name the following
	Distinguished Name will be created:

	    cn=proxyAgent,ou=profile,${namecontext}
	
EOF
    ;;
    namecontext) cat <<EOF

	Naming Context

	The LDAP naming model supports two styles of naming the top
	node of the DIT, or naming context. One is per the original
	X.500 where a country (c=) and organization (o=) are used.
	The other is per RFC 2247 where the domain name is broken up
	into its components (dc=).  By default the Netscape Directory
	Server setup use the X.500 model of (o=).

	The domain name style is popular and recommended here as it
	uses your company's registered Internet Domain Name System
	(DNS) address.

EOF
    ;;
    bindpw) cat <<EOF

	Enter the authentication password for the distinguished name
	(binddn).  

	The password is passed to the ldapsearch(1) and ldapmodify(1)
	commands when retrieving or storing entries on the Directory
	Server.

EOF
    ;;
    sorry) cat <<EOF

	Sorry!  But no help information has been supplied for this question.

EOF
    ;;
    esac

}

## #####################################################################
## check_root_id()
## If user is not root then abort script.
##
function check_root_id {
    case $(/bin/id) in
    uid=0\(root\)*)
		return 0;;
    *)
		print -u2 "Only the super-user should execute this script!"
	        exit 1;;
    esac
}
## #####################################################################
## Get terminal codes to enable/disable bold type.
##
function getTerm {
    case $TERM in
    "sun-cmd") # For some reason 'tput' fails to work for sun-cmd!
	bold='[K[7m'
	norm='[m'
	break;;
    "emacs") # tput complains about emacs
	bold=""
	norm=""
	break;;
    *)
	bold=`tput bold`
	norm=`tput sgr0`
    esac
}
## #####################################################################
## debug(int level, str message, ...)
## If level is <= DEBUG the message is displayed on stderr.
##
function debug {
    [[ $1 -le $DEBUG ]] && print -r -u2 -- "${@:2}"
}
## #####################################################################
## msg(int level, str message, ...)
## Display messages of level VERBOSE or less. logs all to logfile
##
function msg {
    print -r -- "${@:2}" >> $logFile
    [[ $1 -le $VERBOSE ]] && print -- ${@:2}
}
## #####################################################################
## emsg(str message, ...)
## message displayed on terminal and logged to logfile.
function emsg {
    print -r -- "${@}" >> $logFile
    print -ru2 -- "${@}"
}
## #####################################################################
## ask(str question, str default)
## Prompt for answer, use default if no answer supplied.
function ask {
    typeset question="${bold}${1} ${2:+[}${2}${2:+]}${norm}? "
    if [ $noQuery -eq $TRUE -a -n "$2" ]; then
	REPLY=$2
	print $question $REPLY
    else
	read -r "REPLY?${question}" || return 1
	[[ -z $REPLY ]] && REPLY=$2
    fi
    return 0
}
## #####################################################################
## confirm(str question, str default, str helpArg)
## prompts user, expects an answer of Yes, No or Quit.
## 10 Jan 2001 added helpArg.  If set then 'help' can be entered too.
function confirm {
    typeset -l answer
    while :
    do
	ask "${1} [Yes,No,Quit${3:+,?}]" "${2:-yes}" || return 1
	answer=$REPLY
	case $answer in
	y|yes)		return 0;;
	n|no)		return 1;;
	h|help|\?)	showInfo ${3:-sorry};;
	q|quit)		print "Exiting at users request"; exit;;
	*)		print "Please enter Yes, No${3:+, Help} or Quit!";;
	esac
    done
}
## #####################################################################
## getPasswd(str question)
## Prompts for password. Sets REPLY to password
function getPassword {
    /usr/bin/stty -echo
    trap "/usr/bin/stty echo;exit" INT QUIT
    while :
    do
	ask "$1"
	print
	if [ -n "$REPLY" ]; then
	    break;
	fi
	print "You must enter a password!  Use Ctrl-C to exit setup!"
    done
    /usr/bin/stty echo
    trap - INT QUIT
    return
}
## #####################################################################
function updatePasswdScheme {
    STEP=4
    hr
    msg 1 "STEP $STEP ---- Changing Password Storage Scheme to crypt Format"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" $STEP
    do
	msg 2 "Progressing step $STEP"


	typeset -u uptmp=${slapdParams[pw_storagescheme]}
	if [[ $uptmp = "CRYPT" ]]; then
	    msg 1 "Password scheme already set to crypt."
	else
	    while confirm \
		"Change password storage scheme from \
\"${slapdParams[pw_storagescheme]}\" to \"CRYPT\"" "Yes"
	    do
		slapdParams[pw_storagescheme]="CRYPT";
		writeConf $slapdDir/config/slapd.conf pw_storagescheme
		break;
	    done
	fi

    msg 2 "Completed step $STEP"
    break
    done # stage 4
}
## #####################################################################
## getSlapdDir()
## Get slapd configuration directory: sets slapdDir
function getSlapdDir {
    typeset slapdDirs
    ## Get configuration directory.
    while :
    do
	if [ -z $DSHOME ]; then
	    hr
	    print -n "Enter Q to quit or the "
	    print "pathname of Netscape Directory Server 4.x installation"
	    ask "Install Directory" $NSHOME
	    if [[ $REPLY == ?([Qq])*([uit|UIT]) ]]; then
		exit 0
	    fi
	    DSHOME=$REPLY
	fi

	if [ ! -d $DSHOME ]; then
	    emsg "Warning: Directory not found: $DSHOME"
	    DSHOME=""
	    continue ## Ask for the directory again.
	fi

	set -- $DSHOME/slapd-*

	if [[ $# > 1 || $# == 1 && -d $1 ]]; then
	    break ## one or more slapd directories exist.
	else
	    emsg "Error: no slapd configuration directories in $DSHOME"
	    DSHOME=""
	fi
    done
    ## Now find slapd directory; there may be more than one!
    cd $DSHOME
    slapdDirs=(slapd-*) ## file names begining with slapd-* in $PWD
    if [ ${#slapdDirs[@]} -eq 1 ]; then
	slapd=${slapdDirs[0]}
    else
	print "Select the slapd configuration that you wish to configure"
	select slapd in ${slapdDirs[@]} Quit
	do
	    case $slapd in
	    slapd-*)	if [ -d $slapd/. ]; then
			    break
			else
			    emsg "Warning: $slapd is not a directory!"
			fi
			;;
	    Quit)	exit;;
	    *)		print "You must select one of the above.";;
	    esac
	done
    fi

    slapdDir=$DSHOME/$slapd
    msg 3 "using slapd directory: $slapdDir"

}

## #####################################################################
## getNisDomain()
## Asks user to supply domain name: sets nisDomain
function getNisDomain {
    typeset dns nis q
    hr
    [[ $VERBOSE -gt 1 ]] && showInfo "domainame"

    q="Select domain name to be used, Select Other to supply another name:"
    if [ -f /etc/resolv.conf ]; then
	dns=$(/bin/grep -i "^domain[ ]+*" /etc/resolv.conf)
	dns=${dns#* }
    fi
    nis=$(/bin/domainname)

    print $bold$q$norm
    [[ $VERBOSE -gt 1 ]] && showInfo "fqdn"
    select nisDomain in $dns $nis $a Other Quit
    do
	case $nisDomain in
	Quit) 	exit;;
	$dns) 	[[ -n "$dns" ]] && break;;
	$nis) 	[[ -n "$nis" ]] && break;;
	$a)	[[ -n "$a" ]] && break;;
	Other*)
		ask "Enter domain name" "$nis"
		nisDomain=$REPLY
		while confirm "Use $nisDomain for Native LDAP nisDomain"
		do
		    break 2
		done
		;;
	*) "Please select one of the above.";;
	esac
	print $bold$q$norm
    done
    msg 2 "---- Using Domainname: $nisDomain"
}
## #####################################################################
## readConf(pathname filename, params ...)
##
## Reads filename looking for params.
## The associate array slapdParama must be predefined: typeset -A
##
function readConf {
    typeset -Hr filename=$1
    shift
    typeset params="$@"
    typeset nparams=$#
    typeset -i count=0
    msg 3 "Reading configuration file:\n\"$filename\"\nfor parameters: ${@:2}"
    if [[ ! -f $filename ]]; then
	emsg "Fatal: config file does not exist: $filename"
	exit 1
    fi
    busyOn
    exec 3< $filename
    while read -u3
    do
	IFS=' 	' set -- $REPLY
	debug 5 "readConf: $REPLY"
	for item in $params
	do case $item in
	    $1)	slapdParams[$item]=${@:2}
			debug 1 "readConf: $filename: $item"
			debug 2 "= ${slapdParams[$item]}"
			[[ $count -eq $nparams ]] && break 2
			;;
	    esac
	done
    done
    exec 3<&-
    busyOff
}
## #####################################################################
## writeConf(path filename, params...)
##
## Updates filename with changed parameters in slapdParams[params...]
##
function writeConf {
    typeset -Hr filename=$1
    shift
    params="$@"
    debug 2 "Write Params are: $params"
    msg 1 "Preparing config file \"$filename\""
    busyOn
    while read -r
    do
	IFS=' 	' set -- $REPLY
	debug 5 "writeConf: testing: $1"
	for item in $params
	do case $item in
	    $1)
			if [[ ${slapdParams[$item]} != "${@:2}" ]]; then
			    print "${1}\t${slapdParams[$item]}"
			fi
			debug 1 Updated ${1}
			;;
	    *)		print -- "$REPLY"
			;;
	    esac
	done
    done <$filename >/tmp/${filename##*/}.$$
    installFile $filename /tmp/${filename##*/}.$$
    startSlapd
    busyOff
}
## #####################################################################
## writeLdap(...)
##
## executes ldapmodify which expects to find input on the stdin
##
function writeLdap {
    if [[ $VERBOSE -gt 2 || $DEBUG -gt 0 ]]; then
	tee /dev/tty | $LDAPMODIFY $ldapArgs -D "${ldapRootdn//\"/}" -w "$ldapPw" \
	    $@ 2>&1 | /usr/bin/tee -a $logFile
	set +x
    else
	$LDAPMODIFY $ldapArgs -D "${ldapRootdn//\"/}" -w "$ldapPw" \
	    $@ 2>&1 | /usr/bin/tee -a $logFile
    fi
    if [[ -f $rejFile ]]; then
	emsg "writeLdap: ldapmodify rejections:"
	cat $rejFile | tee -a $logFile
	emsg "writeLdap: end ldapmodify rejections."
	rm $rejFile
    fi
}
## #####################################################################
function pingPort {
    ## Note: the open is in a sub shell, and will auto close.
    (exec 4</dev/tcp/127.0.0.1/${1}) 2>/dev/null
    return $?
}
## #####################################################################
## ldapReadBase(int port)
##
## Discovers namingcontexts and supported options.
## But this could be useful for other information too.
##
function ldapReadBase {
    hr
    msg 1 "Step 0 ---- Initialising -- Reading schema base"
    msg 3 "Verifying Directory requirements and discovering Naming Context(s) of DIT"
    typeset -i port=$1
    typeset -i i=0 err
    msg 2 "Probing slapd on port $port for \"base\" objects"
    busyOn
    $ldapSearch -b "" -p $port -s base "objectclass=*" |
    while read -r -t5
    do
	debug 4 "probeLdapBase: $REPLY"
	case $REPLY in
	namingcontexts=o=NetscapeRoot)
				: ## Not interested in this one.
				;;
	namingcontexts*)	debug 3 "probeLdapBase: found: \"$REPLY\""
				let i=i+1
				namingContexts[$i]=${REPLY#*=}
				;;
	*=2.16.840.1.113730.3.4.9)
				srvVLVsupport=$TRUE
				debug 3 "probeLdapBase: vlv indexing supported"
				;;
	*=1.2.840.113556.1.4.473)
				srvSort=$TRUE
				debug 3 \
				    "probeLdapBase: Server Sorting supported"
				;;
	*=2.16.840.1.113730.3.4.2)
				srvSimplePageMode=$TRUE
				debug 3 \
				    "probeLdapBase: Simple page mode supported"
				;;
	esac
    done
    busyOff
    if [[ $srvVLVsupport -ne $TRUE ]]; then
	emsg "Warning: No Support for Virtual List Views!"
    fi

    if [[ $srvSimplePageMode -ne $TRUE ]]; then
	emseg "Warning: No Support for Simple Page Mode!"
    fi

    if [ ${#namingContexts[@]} -le 0 ]
	then
	    emsg "Error: No namingcontexts found!"
	    emsg "Is LDAP installed correctly?"
	    emsg "Sometimes this occurs because slapd was not ready!"
	    exit 1;
	else if [ ${#namingContexts[@]} -eq 1 ]
	    then
		namecontext=${namingContexts[@]}
	else
	    typeset name
	    print "\n\nMore than one namingcontext is present.  Please"
	    print "select namingcontext that will be used for Native LDAP I:"
	    select namecontext in ${namingContexts[@]} Quit
	    do
		for name in ${namingContexts[@]}
		do
		    if [ $name == $namecontext ]; then
			break 2
		    fi
		done
		case $namecontext in
		Quit)
			while confirm "Really quit" "No"
			do
			    msg 1 "Exiting at users request."
			    exit 1
			done
			;;
		*)	print -u2 "You must select one of the above!";;
		esac
	    done
	fi
    fi

    ## Check!
    if [[ $namecontext != dc=* ]]; then
	emsg 'Warning: expected to find namecontext of the form'
	emsg "\"dc=sun,dc=com\" as recommended in RFC 2247:"
	emsg "    http://www.rfc-editor.org/rfc/rfc2247.txt"
	emsg "Please confirm that you want to use namecontext \"$namecontext\""
	while ! confirm "Use namecontext $namecontext" "No" "namecontext"
	do
	    emsg "Please refer to the LDAP setup and configuration guide for"\
	         " for further information."
	    exit 1
	done
    fi

}
## #####################################################################
## updateSlapdOcConf(path slapdDir)
##
## Updates slapd.oc.conf as per LDAP Setup and Configuration Guide
##
function updateSlapdOcConf {
    typeset filename=$1/config/slapd.oc.conf
    typeset tmp=/tmp/slapd.oc.conf.$$
    typeset ofs=$IFS

    STEP=2
    hr
    msg 1 "STEP $STEP ---- Modify slapd.oc.conf"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" "$STEP"
    do
    msg 2 "Progressing step $STEP"
    if [[ ! -f $filename ]]; then
	emsg "Fatal: config file does not exist: $filename"
	exit 1
    fi
    busyOn
    exec 5< $filename  ## open fd
    IFS=""
    while read -ru5
    do
	case $REPLY in
	objectclass+([\t ])ipNetwork)
	    print -- "$REPLY"
	    while read -ru5
	    do
		case $REPLY in
		"") 			print -- $REPLY; break;;
		*ipNetworkNumber,)	print -- $"\t\tipNetworkNumber";;
		*cn?([,])) 		: ;;
		*allows) 		print -- "$REPLY"; print $"\t\tcn,";;
		*) 			print -- "$REPLY";;
		esac
	    done
	    ;;
	*) print -- "$REPLY"
	    ;;
	esac
    done > $tmp
    exec 5<&- ## Close fd
    IFS=$ofs
    busyOff
    installFile $filename $tmp

    msg 2 "Completed step $STEP"
    break
    done # while confirm
}
## #####################################################################
## isnum(str num)
##
## Confirms that str is numeric.
## taken The New Kornshell Command and Programming language
##
function isnum {
    case $1 in
    ?([-+])+([0-9])?(.)*([0-9])?([Ee]?([-+])+([0-9])) )
	return 0;;
    ?([-+])*([0-9])?(.)+([0-9])?([Ee]?([-+])+([0-9])) )
	return 0;;
    *) return 1;;
    esac
    # not reached
}

## #####################################################################
## stopSlapd
##
function stopSlapd {
    msg 3 "Stopping slapd"
    [[ $SAFEMODE -eq TRUE ]] && return 0
    while pingPort ${slapdParams[port]}
    do
	busyOn
	$slapdDir/stop-slapd
	busyOff
	return
    done
    msg 4 "slapd not running."
}
## #####################################################################
## StartSlapd
##
function startSlapd {
    typeset -i code=0
    msg 3 "Starting slapd"
    while ! pingPort ${slapdParams[port]}
    do
	busyOn
	$slapdDir/start-slapd || code=$?
	if [ $code -eq 0 ]; then
	    slapdProc=$!
	    sleep 5
	fi
	busyOff
	return $code
    done
    msg 4 "slapd already started"
}

## #####################################################################
function busyOn {
    [[ $VERBOSE > 2 || $DEBUG > 0 || $BUSYID > 0 ]] && return
    trap "kill $BUSYID;exit" TERM KILL INT
    (trap "end=1" TERM
	end=0
	while [ $end -eq 0 ]
	    do
	    for c in '\' '|' '/' '-'
		do print -n -- $c; sleep 1; print -n "\b"
		[[ $end -ne 0 ]] && break
	    done
	done
    )&
    BUSYID=$!
}
## #####################################################################
function busyOff {
    [[ $VERBOSE > 2 || $DEBUG > 0 || $BUSYID == 0 ]] && return
    trap - TERM KILL INT
    kill $BUSYID
    debug 3 "Waiting for $BUSYID to finnish"
    wait $BUSYID
    BUSYID=0
}
## #####################################################################
## installFile(path orig, path new)
##
## Compares orig with new: backing up and installing where necessery
##
function installFile {
    if [ ! -f $2 ]; then
	[[ $SAFEMODE -eq TRUE ]] && return 0
	emsg "Warning: file \"$2\" Not found!"
	return 1
    fi
    if [ ! -f $1 ]; then
	[[ $SAFEMODE -eq TRUE ]] && return 0
	emsg "Warning: file \"$1\" Not found!"
	return 1
    fi
    if [ ! -s $2 ]; then
	emsg "Warning: $2 is an empty file!  Will not install."
	return 1
    fi
    diff -b $1 $2 >/dev/null
    if [ $? -eq 0 ]; then
	msg 3  "No modifications in $2 for $1"
	rm $2
    else
	stopSlapd
	msg 3 "Applying modifications to $1"
	msg 4 "	moving original file to $1.$SUF"
	run mv $1 $1.$SUF
	msg 4 "	Creating new $1"
	run mv $2 $1
	while ! startSlapd
	do
	    emsg "Error: Slapd failed to start after modification!"
	    emsg "Notice: Installing previous file: $1.$SUF"
	    emsg "Notice: Erroneous file saved as $1.FAIL"
	    run mv $1 $1.FAIL
	    run cp $1.$SUF $1
	    emsg "You will need to modify ${1##*/} by hand."
	    while ! confirm "Continue" "Yes"
	    do
		exit 1
	    done
	    return 1
	done
    fi
    return 0
}
## #####################################################################
## change_aci(string namecontext, int port)
##
## Converts ACI for Native LDAP I
function change_aci {
    typeset suffix=$1
    typeset port=$2
    typeset -i i=0 replace=0
    typeset tmpfile=/tmp/native_aci.$$
    # These searches (old, new) may need some work!
    typeset old=$"*ldap:///self*"
    typeset new='aci=(targetattr!="cn || uid || uidNumber || gidNumber || homeDirectory || loginShell || gecos || shadowLastChange || shadowMin || shadowMax ||shadowWarning || shadowInactive || shadowExpire || shadowFlag || memberUid")(version 3.0; acl "Allow self entry modification"; allow(write) userdn = "ldap:///self"; )'

    STEP=6
    hr
    msg 1 "STEP $STEP ---- Modifying Self-Entry Modification"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" "$STEP"
    do
	msg 2 "Progressing step $STEP"

	startSlapd $slapdDir
	[[ $DEBUG -gt 9 ]] && set -x
	${ldapSearch} -p $port -s base -b "$suffix" "aci=*" aci |
	while read -r -t2
	do
	    debug 4 "change_aci: $REPLY"
	    case $REPLY in
	    $new)
			    ## Appears to be already set!
			    debug 1 "New aci already installed!"
			    let replace=replace+2
			    ;;
	    $old)
			    ## This is the one we dont want!
			    debug 2 "change_aci: replacing: $REPLY"
			    let i=i+1
			    aci[$i]=$new
			    let replace=replace+1
			    ;;
	    aci*)
			    let i=i+1
			    aci[$i]=$REPLY
			    ;;
	    esac
	done
	[[ $DEBUG -gt 9 ]] && set +x
	case $replace in
	0)	emsg "Error: Failed to find original ACI in \"$suffix\"!"
	    emsg "${aci[@]}"
	    emsg "Please update ACI for ${ldapRootdn} manually!"
	    ;;
	1)
	    ## Write out new aci to ascii file
	    exec 3<> $tmpfile

	    print -u3 "dn: $suffix"
	    print -u3 "changetype: modify"
	    print -u3 "replace: aci"
	    i=1
	    while [ $i -le  ${#aci[@]} ]
	    do
		print -u3 -r "aci: ${aci[$i]#*=}"
		let i=i+1
	    done
	    ## close file
	    exec 3<&-
	    writeLdap -f $tmpfile
	    run rm $tmpfile
	    ;;
	2)  emsg "Notice: ACI already modified." ;;
	3)  emsg "Error: Both old and new ACI exist in $suffix!" ;;
	*)  emsg "Fatal: Unexpected value of $replace in \"replace\"!"
	    exit;;
	esac

	msg 2 "Completed step $STEP"
	break
    done # confirm modify ACI

}
## #####################################################################
function ldapAddVLVaci {

    STEP=7
    hr
    msg 1 "STEP $STEP ---- Setting VLV Control ACI"
    if [ $srvVLVsupport -eq $FALSE ]; then
	msg 1 "Note: Server does not supplort Virtual List View (VLV) control"
	return
    fi
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" "$STEP"
        do
	msg 2 "Progressing step $STEP"


	writeLdap -a -c <<-EOS
	dn: oid=2.16.840.1.113730.3.4.9,cn=features,cn=config
	changetype: modify
	replace: aci
	aci: (targetattr != "aci")
	 (version 3.0; acl "VLV Request Control"; allow(read,search,compare)
	 userdn = "ldap:///anyone";)


	EOS
	msg 2 "Completed step $STEP"

	break
    done # confirm add VLV aci
}

## #####################################################################
## ldapAddAttr()
##
## add attribute definitions for native-ldap
## LDIF Provided by Michael Haines
## #####################################################################
function ldapAddAttr {
    STEP=1
    hr
    msg 1 "STEP $STEP ---- Add Schema attributes for native support"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" "$STEP"
    do
	msg 2 "Progressing step $STEP"


    writeLdap -a -c <<-EOS
	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.1.1.1.28 NAME 'nisPublickey'
	  DESC 'NIS public key' SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.1.1.1.29 NAME 'nisSecretkey'
	  DESC 'NIS secret key' SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.1.1.1.30 NAME 'nisDomain'
	  DESC 'NIS domain' SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.1.1.12 NAME 'nisNetIdUser'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.1.1.13 NAME 'nisNetIdGroup'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.1.1.14 NAME 'nisNetIdHost'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( rfc822mailMember-oid NAME 'rfc822mailMember'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 2.16.840.1.113730.3.1.30 NAME 'mgrpRFC822MailMember'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.15 NAME 'SolarisLDAPServers'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.16 NAME 'SolarisSearchBaseDN'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.12' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.17 NAME 'SolarisCacheTTL'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.18 NAME 'SolarisBindDN'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.12' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.19 NAME 'SolarisBindPassword'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.20 NAME 'SolarisAuthMethod'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15')

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.21
	  NAME 'SolarisTransportSecurity'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15')

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.22
	  NAME 'SolarisCertificatePath'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.23
	  NAME 'SolarisCertificatePassword'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.24
	  NAME 'SolarisDataSearchDN'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15')

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.25
	  NAME 'SolarisSearchScope'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.26
	  NAME 'SolarisSearchTimeLimit'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.27' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.27
	  NAME 'SolarisPreferredServer'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15')

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.28
	  NAME 'SolarisPreferredServerOnly'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.29
	  NAME 'SolarisSearchReferral'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.4
	  NAME 'SolarisAttrKeyValue'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.5
	  NAME 'SolarisAuditAlways'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.6
	  NAME 'SolarisAuditNever'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.7
	  NAME 'SolarisAttrShortDesc'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.8
	  NAME 'SolarisAttrLongDesc'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.9
	  NAME 'SolarisKernelSecurityPolicy'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.10
	  NAME 'SolarisProfileType'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.11
	  NAME 'SolarisProfileId'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.12
	  NAME 'SolarisUserQualifier'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.13
	  NAME 'SolarisAttrReserved1'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.14
	  NAME 'SolarisAttrReserved2'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.15' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.1 NAME 'SolarisProjectID'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.27' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.2 NAME 'SolarisProjectName'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' SINGLE-VALUE )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( 1.3.6.1.4.1.42.2.27.5.1.3 NAME 'SolarisProjectAttr'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' )

	dn: cn=schema
	changetype: modify
	add: attributetypes
	attributetypes: ( memberGid-oid NAME 'memberGid'
	  SYNTAX '1.3.6.1.4.1.1466.115.121.1.26' )
	EOS
	msg 2 "Completed step $STEP"
	break
    done
}
## #####################################################################
##
## objectclass definitions for native-ldap
## LDIF Provided by Michael Haines
##
function ldapAddObj {

    STEP=3
    hr
    msg 1 "STEP $STEP ---- Adding Schema Objects for native support"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" "$STEP"
    do
	msg 2 "Progressing step $STEP"


    writeLdap -a -c <<-EOS
	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.1.1.2.14 NAME 'NisKeyObject'
	  SUP 'top' MUST (objectclass $ cn $ nisPublickey $ nisSecretkey)
	  MAY (uidNumber $ description))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.1.1.2.15 NAME 'nisDomainObject'
	  SUP 'top' MUST (objectclass $ nisDomain))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.4.1.42.2.27.5.2.7 NAME 'SolarisNamingProfile'
	  SUP 'top' MUST (objectclass $ cn $ SolarisLDAPservers $
	  SolarisSearchBaseDN) MAY (SolarisBindDN $ SolarisBindPassword $
	  SolarisAuthMethod $ SolarisTransportSecurity $
	  SolarisCertificatePath $ SolarisCertificatePassword $
	  SolarisDataSearchDN $ SolarisSearchScope $
	  SolarisSearchTimeLimit $ SolarisPreferredServer $
	  SolarisPreferredServerOnly $ SolarisCacheTTL $
	  SolarisSearchReferral))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 2.16.840.1.113730.3.2.4 NAME 'mailGroup' SUP
	  'top' MUST (objectclass $ mail) MAY (cn $ mgrpRFC822MailMember))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.4.1.42.2.27.1.2.5 NAME 'nisMailAlias'
	  SUP 'top' MUST (objectclass $ cn) MAY (rfc822mailMember))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.4.1.42.2.27.1.2.6 NAME 'nisNetId'
	  SUP 'top' MUST (objectclass $ cn) MAY (nisNetIdUser $
	  nisNetIdGroup $ nisNetIdHost))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.4.1.42.2.27.5.2.2 NAME 'SolarisAuditUser'
	  SUP 'top' MUST (objectclass) MAY (SolarisAuditAlways $
	  SolarisAuditNever))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.4.1.42.2.27.5.2.3 NAME 'SolarisUserAttr'
	  SUP 'top' MUST (objectclass) MAY (SolarisUserQualifier $
	  SolarisAttrReserved1 $ SolarisAttrReserved2 $ SolarisAttrKeyValue))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.4.1.42.2.27.5.2.4 NAME 'SolarisAuthAttr'
	  SUP 'top' MUST (objectclass $ cn) MAY (SolarisAttrReserved1 $
	  SolarisAttrReserved2 $ SolarisAttrShortDesc $
	  SolarisAttrLongDesc $ SolarisAttrKeyValue))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.4.1.42.2.27.5.2.5 NAME 'SolarisProfAttr'
	  SUP 'top' MUST (objectclass $ cn) MAY (SolarisAttrReserved1 $
	  SolarisAttrReserved2 $ SolarisAttrLongDesc $ SolarisAttrKeyValue))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.4.1.42.2.27.5.2.6 NAME 'SolarisExecAttr'
	  SUP 'top' MUST (objectclass) MAY (SolarisKernelSecurityPolicy $
	  SolarisProfileType $ SolarisAttrReserved1 $ SolarisAttrReserved2 $
	  SolarisProfileID $ SolarisAttrKeyValue))

	dn: cn=schema
	changetype: modify
	add: objectclasses
	objectclasses: ( 1.3.6.1.4.1.42.2.27.5.2.1 NAME 'SolarisProject'
	  SUP 'top' MUST (objectclass $ SolarisProjectID $ SolarisProjectName)
	  MAY (memberUid $ memberGid $ description $ SolarisProjectAttr))

	EOS

	msg 2 "Completed step $STEP"
	break
	done
}

## #####################################################################
## ldapAddIndex
##
function ldapAddIndex {

    STEP="11"
    hr
    msg 1 "STEP $STEP ---- Creating Indexes"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP - create indexes" \
	"Yes" "$STEP"
    do
	msg 2 "Progressing step $STEP"

	stopSlapd
	msg 1 "-- Creating Indexes"
	for index in membernisnetgroup nisnetgrouptriple
	do
	    db2index $index pres,eq,sub
	done

	for index in memberuid \
		    macAddress uid uidNumber gidNumber ipHostNumber \
		    ipNetworkNumber ipProtocolNumber oncRpcNumber \
		    ipServiceProtocol ipServicePort nisDomain \
		    nisMapName mail
	do
	    db2index $index pres,eq
	done
	msg 1 "Finnished creating indexes"
	startSlapd
#	msg 0 "Showing indexes"
#	$ldapSearch -s base -b "" objectclass=\* vlvsearch | tee -a $logFile
	msg 2 "Completed step $STEP"
	break
    done # step 11
}
function db2index {
    typeset pwd name=$1 type=$2
    msg 1 "Creating index for attribute $name of type $type"
    pwd=$PWD
    cd $DSHOME/bin/slapd/server
    run ./ns-slapd db2index -f $slapdDir/config/slapd.conf -t ${name}:${type}
    cd $pwd
    print
}
## #####################################################################
## ldapAddTopClasses(rDN, NisDomainName)
##
function ldapAddTopClasses {

    STEP=5
    hr
    msg 1 "STEP $STEP ---- Adding New Containers (top-level classes)"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" "$STEP"
    do
	msg 2 "Progressing step $STEP"


## For some reason this next tiny bit of LDIF fails if using the netscape
## ldapmodify command.  And yet works just fine if using /bin/ldapmodify!
    writeLdap <<-EOS
	dn: ${1}
	changetype: modify
	objectclass: nisDomainObject
	nisdomain: ${2}


	EOS

#    writeLdap -a -c <<-EOS
#	dn: ${1}
#	objectClass: top
#	objectClass: domain
#
#	dn: ou=people,${1}
#	ou: People
#	objectClass: top
#	objectClass: organizationalUnit
#
#
#	EOS

     writeLdap -a -c <<-EOS
	dn: ou=Group,${1}
	ou: Group
	objectClass: top
	objectClass: organizationalUnit

	dn: ou=rpc,${1}
	ou: rpc
	objectClass: top
	objectClass: organizationalUnit

	dn: ou=protocols,${1}
	ou: protocols
	objectClass: top
	objectClass: organizationalUnit

	dn: ou=networks,${1}
	ou: networks
	objectClass: top
	objectClass: organizationalUnit

	dn: ou=netgroup,${1}
	ou: netgroup
	objectClass: top
	objectClass: organizationalUnit

	dn: ou=aliases,${1}
	ou: aliases
	objectClass: top
	objectClass: organizationalUnit

	dn: ou=Hosts,${1}
	ou: Hosts
	objectClass: top
	objectClass: organizationalUnit

	dn: ou=services,${1}
	ou: services
	objectClass: top
	objectClass: organizationalUnit

	dn: ou=Ethers,${1}
	ou: Ethers
	objectClass: top
	objectClass: organizationalUnit

	dn: ou=profile,${1}
	ou: profile
	objectClass: top
	objectClass: organizationalUnit


	EOS
	msg 2 "Completed step $STEP"
	print
	STEP="5.1"
	msg 1 "Step $STEP ---- Add default automount containers"
	[[ $VERBOSE -gt 1 ]] && showInfo $STEP
	while confirm "OK to progress step $STEP - add automount containers"\
		"Yes" "$STEP"
	do
	    msg 2 "Progressing step $STEP"

	    writeLdap -a -c <<-EOS
		dn: nismapname=auto_home,${1}
		nismapname: auto_home
		objectClass: top
		objectClass: nisMap

		dn: nismapname=auto_direct,${1}
		nismapname: auto_direct
		objectClass: top
		objectClass: nisMap

		dn: nismapname=auto_master,${1}
		nismapname: auto_master
		objectClass: top
		objectClass: nisMap


	EOS

	msg 2 "Completed step $STEP"
	break
	done # step 5.1: confirm automount

    break
    done # step 5: confirm containers

}
## #####################################################################
function ldapAddVlvIndex {
    typeset name=$1
    typeset entry=$2
    typeset filter=$3
    typeset -i vlvScope=1 # 0=Base search 1=One Level, 2=Sub Tree
    writeLdap -a -c <<-EOS
	dn: cn=${name},cn=config,cn=ldbm
	objectclass: top
	objectclass: vlvSearch
	cn: $name
	vlvBase: ou=${entry},${namecontext}
	vlvScope: $vlvScope
	vlvFilter: (objectclass=${filter})
	aci: (target="ldap:///cn=${name},cn=config,cn=ldbm")(targetattr="*")
	 (version 3.0; acl "Config";allow(read,search,compare)
	 userdn="ldap:///anyone";)

	dn: cn=${name},cn=${name},cn=config,cn=ldbm
	cn: ${name}
	vlvSort: cn uid
	objectclass: top
	objectclass: vlvIndex

	EOS

}
## #####################################################################
## function ldapAddVlv()
##
## #####################################################################
function ldapAddVlv {
    STEP=12
    hr
    msg 1 "STEP $STEP ---- Adding Virtual List View (VLV) Indexes"
    if [ $srvVLVsupport -ne $TRUE ]; then
	msg 1 "Note: Server does not supplort Virtual List View (VLV) control"
	return
    fi
    
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" "$STEP" "$STEP"
    do
	msg 2 "Progressing step $STEP"

    
	ldapAddVlvIndex "getpwent"      "people"    "posixAccount"
	ldapAddVlvIndex "getspent"      "people"    "posixAccount"
	ldapAddVlvIndex "getgrent"      "group"     "posixGroup"
	ldapAddVlvIndex "gethostent"    "hosts"     "ipHost"
	ldapAddVlvIndex "getnetent"     "networks"  "ipNetwork"
	ldapAddVlvIndex "getprotoent"   "protocols" "ipProtocol"
	ldapAddVlvIndex "rpcent"        "rpc"       "oncRpc"
	ldapAddVlvIndex "getaliasent"   "aliases"   "rfc822MailGroup"
	ldapAddVlvIndex "getserviceent" "services"  "ipService"
        msg 2 "Completed step $STEP"
	break
    done # Step 12: add VLV indexes

    STEP="12.1"
    hr
    msg 1 "STEP $STEP ---- Creating Virtual List View (VLV) Indexes"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP - create VLV indexes" \
	"Yes" "$STEP"
    do
	msg 2 "Progressing step $STEP"

	stopSlapd
	msg 1 "-- Creating Virtual List View Indexes"
	for index in getpwent getspent getgrent gethostent getnetent \
		     getprotoent rpcent getaliasent getserviceent
	do
	    msg 1 $index
	    run $slapdDir/vlvindex $index
	    print
	done
	msg 1 "Finnished creating VLV indexes"
	startSlapd
	msg 0 "Showing vlv indexes"
	$ldapSearch -s base -b "" objectclass=\* vlvsearch | tee -a $logFile
	msg 2 "Completed step $STEP"
	break
    done # step 12.1
}

## #####################################################################
## ldapReadCache(int port)
##
## Discovers: nsslapd-cachesize nsslapd-lookthroughlimit nsslapd-dbcachesize
##
function ldapReadCache {
    $ldapSearch -p ${slapdParams[port]} -D "${ldapRootdn}" -w "$ldapPw" -b \
	"cn=config,cn=ldbm"  "(&(objectclass=extensibleObject)(cn=config))" \
	nsslapd-cachesize nsslapd-lookthroughlimit nsslapd-dbcachesize |
    while read -r -t 2
    do
	debug 4 "ldapreadCache: $REPLY"
	case $REPLY in
	nsslapd-cachesize=*)
	    slapdParams[cachesize]=${REPLY#*=};;
	nsslapd-lookthroughlimit=*)
	    slapdParams[lookthroughlimit]=${REPLY#*=};;
	nsslapd-dbcachesize=*)
	    slapdParams[dbcachesize]=${REPLY#*=};;
	esac
    done
}
## #####################################################################
function hr {
print "
_______________________________________________________________________________

"
}
## #####################################################################
function showCacheValues {
    print
    msg 0 "\tlookthroughlimit: ${slapdParams[lookthroughlimit]}"
    msg 0 "\tcachesize: ${slapdParams[cachesize]}"
    msg 0 "\tdbcachesize: ${slapdParams[dbcachesize]}"
}
## #####################################################################
function ldapWriteCache {
#    $LDAPMODIFY -c $ldapArgs -D "${ldapRootdn}" -w "$ldapPw" <<-EOS
    writeLdap -c <<-EOS
	dn: cn=config,cn=ldbm
	changetype: modify
	replace: nsslapd-lookthroughlimit:
	nsslapd-lookthroughlimit: ${slapdParams[lookthroughlimit]}
	-
	replace: nsslapd-cachesize:
	nsslapd-cachesize: ${slapdParams[cachesize]}
	-
	replace: nsslapd-dbcachesize:
	nsslapd-dbcachesize: ${slapdParams[dbcachesize]}


	EOS
}
## #####################################################################
function calcCache {
    ## Only do this once...
    [[ ${slapdParams[CALClookthroughlimit]} -eq -1 ]] && return
    
## This was an idea I had...  But it seems to be too greedy
## So for now I stick with what I was told on Netscape admin course.

##    pagesize=$(print "pagesize/D"|adb -k)
##    pagesize=${pagesize##*	}
##    case $(isainfo -b) in
##    32)	physmem=$(print "physmem/D"|adb -k);;
##    64)	physmem=$(print "physmem/E"|adb -k);;
##    *) emsg "Error: Unexpected number of bits!";;
##    esac
##    physmem=${physmem##*	}
##    
##    typeset -i memory
##    memory=physmem*pagesize
##    memory=memory/1024
##
##debug 1 "$physmem Bytes x $pagesize Bytes = $memory KB (approx)"

    set x1 x2 x3 x4 freemem rest

    ## So, on the Netscape course they said to use the freemem counter
    ## from the first line!  I know, I was surprised too!
    /usr/bin/vmstat | 
    while read -r -t 2 x1 x2 x3 x4 freemem rest
    do
	while isnum $freemem
	do
	    msg 3 "vmstat shows average freemem of $freemem"
	    if [ $freemem -ge 1600000 ]; then
		## The 4.1x directory server can not use more than 2 GB of
		## of memory (including 25% overhead).  Hence the limit here.
		msg 3 "freemem > 1600000 (1.6gB). adjusted accordingly."
		memory=1600000
	    else
		memory=$freemem
	    fi
	    break
	done
    done

    ## There should not be a look through limit!
    slapdParams[CALClookthroughlimit]=-1

    while isnum $memory
    do 
	typeset -i dbCache entryCache
	typeset -F f

	f=memory*0.75	# Total DB cache
	f=f/1.25	# Cache memory overhead
	f=f*1000	# bytes
	dbCache=$f
	slapdParams[CALCdbcachesize]=$dbCache

	f=memory*0.25	# Total Entry cache
	f=f/1.25	# Cache memory overhead (AEC)
	f=f/0.5	# AEC / Average cache Size
	entryCache=$f
	slapdParams[CALCcachesize]=$entryCache
    break
    done
    unset x1 x2 x3 x4 freemem rest
}

## #####################################################################
function updateCache {
    ldapReadCache
    calcCache

    typeset param change=$FALSE
    hr
    
    [[ $VERBOSE -gt 1 ]] && showInfo "cache"
    while confirm "Change caching parameters" "No" "cache"
    do
	print "Current cache values are:"
	showCacheValues
	print "\nEnter new cache values below:\n"
	typeset -A cache
	for param in lookthroughlimit cachesize dbcachesize
	do
	    ask "	Change $param from ${slapdParams[$param]}" \
		${slapdParams[CALC$param]:-${slapdParams[$param]}}
	    while isnum "$REPLY"
	    do
		if [ ${slapdParams[$param]} -ne $REPLY ]; then
		    slapdParams[$param]=$REPLY
		    change=$TRUE
		fi
		break
	    done
	done
	if [ $change -eq $TRUE ]; then
	    msg 0 "New caching values supplied are:"
	    showCacheValues
	    while confirm \
		"Are you sure you want to apply these cache changes" "yes"
	    do
		change=$FALSE
		ldapWriteCache
		break
	    done
	fi
	ldapReadCache
	showCacheValues
    done

}
## #####################################################################
## proxy agent
## 
## I had proxyagent stuff in line.  but to keep the STEP numbers in line
## I have split them out...  Hence I have to keep checking for the 
## proxyagent name and password.
function getProxyName {
    while [ -z "$proxyAgent" ]
    do
	ask "Enter name to be assigned to the proxy-agent" \
	    "proxyagent" "proxyagent"
	proxyAgent=$REPLY
    done
}
function getProxyPasswd {
    while [ -z "$proxyPw" ] ## Passwords are not the same...
    do
	getPassword "Enter new password for proxyagent \"$proxyAgent\""
	proxyPw=$REPLY
	getPassword "Re-Enter progagent password for confirmation"
	if [ $REPLY != $proxyPw ]; then
	    proxyPw=""
	    print -u2 "Passwords do not match!  Please re-enter"
	 fi
    done
}
## #####################################################################
function ldapAddProxyAgent {

    STEP=8
    hr
    msg 1 "STEP $STEP ---- Adding Proxy Agent Entry"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" "$STEP"
    do
	msg 2 "Progressing step $STEP"

	[[ -z $proxyAgent ]] && getProxyName
	[[ -z $proxyPw ]] && getProxyPasswd

	writeLdap -a -c <<-EOS
	dn: cn=${proxyAgent},ou=profile,${namecontext}
	cn: ${proxyAgent}
	sn: ${proxyAgent}
	objectclass: top
	objectclass: person
	userpassword: ${proxyPw}
	EOS
	msg 2 "Completed step $STEP"
	break
    done # Step 8

    STEP=9
    hr
    msg 1 "STEP $STEP ---- Setting Password Read Permission for proyagent"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress step $STEP" "Yes" "$STEP"
    do
	msg 2 "Progressing step $STEP"


	[[ -z $proxyAgent ]] && getProxyName
	[[ -z $proxyPw ]] && getProxyPasswd

	writeLdap -a -c <<-EOS
	dn: ${namecontext}
	changetype: modify
	add: aci
	aci: (target="ldap:///${namecontext}") (targetattr="userPassword")
	 (version 3.0; acl "password read"; allow (compare,read,search)
	 userdn = "ldap:///cn=${proxyAgent},ou=profile,${namecontext}"; )


	EOS
    msg 2 "Completed step $STEP"
    break;
    done # step 9

    STEP=10
    hr
    msg 1 "STEP 10 ---- Generating the Client Profile"
    [[ $VERBOSE -gt 1 ]] && showInfo $STEP
    while confirm "OK to progress STEP $STEP - create client profile"\
	"Yes" "$STEP"
    do
	msg 2 "Progressing step $STEP"

	[[ -z $proxyAgent ]] && getProxyName
	[[ -z $proxyPw ]] && getProxyPasswd

	ask "Enter name for profile" "${profileName}"
	profileName=$REPLY
	tmp=/tmp/native_gp$$.ldif
	host=$(/bin/uname -n)
	localhost=$(getent hosts $host)
	localhost=${localhost%%?${host}*}:${slapdParams[port]}
	proxyAgentDN="cn=${proxyAgent},ou=profile,${namecontext}"
	/usr/sbin/ldap_gen_profile -P "$profileName" -b $namecontext \
	    -d $nisDomain -a simple \
	    -D "${proxyAgentDN:-$ldapRootdn}" -w "${proxyPw:-$ldapPw}" \
	    $localhost |
	    sed -e 's/^[ 	]//' > $tmp 
	    ## sed above removes leading tab Bug 4346889
	/bin/cat $tmp
	print
	while confirm "Add profile \"$profileName\" as shown above" \
			"Yes" "addprof"
	do
	    msg 1 "Adding profile \"$profileName\""
	    cat $tmp >> $logFile
	    writeLdap -a -f $tmp
	    rm $tmp
	    print
	    msg 1 "Use the following command to setup clients " \
		    "to use this profile:"
	    msg 1 "	ldapclient -P $profileName -d $nisDomain $localhost"
	    print
	    break
	done # Add profile

    msg 2 "Completed step $STEP"
    break
    done # Step 10

}
## #####################################################################
function run {
    if [ ${SAFEMODE} -eq $TRUE ]; then
	print -- SAFE: "$@"
	return 0
    else
	[[ ${DEBUG} -gt 0 ]] && print -- "$@"
	eval $@
	return $?
    fi
   return
}

## Main ####################################################################

NSHOME=${NSHOME:-/usr/netscape/server4}
SUF=$(date +%y%m%d)

integer DEBUG=0 VERBOSE=2 BUSYID=0
typeset -ir FALSE=1 TRUE=0
typeset -i SAFEMODE=$TRUE

typeset ldapSearch="/bin/ldapsearch"
typeset ldapArgs="-n" ## No Updates by default.
typeset logFile=/var/tmp/native_ldap.log
typeset profileName="SolarisNative"
integer noQuery=$FALSE
PS3="Please pick one of the above (return to redisplay list): "

## Hack around getopts to make -? work..
options=':hxqVv#?[level]d#?[level]]c:[path]w:[passwd]D:[binddn]l:[log]n:[nisDomain]a:[proxyagent]A:[proxypw]p:[profile]'
if [[ $* == *-[?h]* ]] ;then
    args="$@"
    print
    set -- -?
    while getopts $options opt
    do :
    done
    [[ $args == *-h ]] && showInfo "help"
    exit 0
fi

## No -? on command line:
while getopts -a ${0##*/} $options opt
do
    case $opt in
    a)	proxyAgent=${OPTARG} ;;
    A)	proxyPw=${OPTARG} ;;
    x)	SAFEMODE=$FALSE
	ldapArgs=${ldapArgs#-n} ## remove "-n" option.
	;;
    v) 	VERBOSE=${OPTARG:-1};;
    V)  ldapArgs="$ldapArgs -v";;
    d) 	DEBUG=${OPTARG:-9} ;;
    c)	DSHOME=$OPTARG ;;
    w)	ldapPw=$OPTARG ;;
    D)	ldapRootdn=$OPTARG ;;
    l)	logFile=${OPTARG} ;;
    n)	nisDomain=${OPTARG} ;;
    p)	profileName=${OPTARG} ;;
    q)	noQuery=$TRUE ;;
    *) 	print -u2 "Invalid option \"$opt\".  Use -h for a list of options."
	exit 1 ;;
    esac
done

check_root_id ## Only run as root.
getTerm       ## Sets globals 'bold' and 'norm' for screen highlighting

print -- "-- $(date) --" >> $logFile ## Announce our selves.
print
msg 0 "Solaris Native LDAP I setup for Netscape Directory Services 4.x"
msg 0 "Script version 1.11 written by Stacey Marshall and Michael Haines."

if [ -a $(/bin/isainfo -b) -ne 32 ]; then
    emsg "Solaris running in $(/bin/isainfo -b) bit may not be supported?"
    while ! confirm "OK to continue" "No" "not32bit"
    do
	exit
    done
fi

if [ $SAFEMODE -eq $TRUE ]; then
    msg 0 "\t------- SAFE MODE -- No Updates will occur -- SAFE MODE ------- "
    [[ $VERBOSE -gt 1 ]] && showInfo "safemode"
    while ! confirm "OK to continue" "Yes" "safemode"
    do
	exit
    done
fi

if [ $VERBOSE -gt 1 ]; then
    showInfo "Welcome"
    while ! confirm "OK to continue" "Yes" "Welcome"
    do
	exit
    done
fi

getSlapdDir ## sets DSHOME and slapdDir

#LDAPMODIFY=$DSHOME/shared/bin/ldapmodify
#rejFile=/tmp/rejfile.$$
#ldapArgs="$ldapArgs -e $rejFile"
#export LD_LIBRARY_PATH=$DSHOME/lib
LDAPMODIFY="/bin/ldapmodify"

if [ ! -f $LDAPMODIFY ]; then
    emsg "Fatal: Not Found: $LDAPMODIFY"
    exit 1
fi

typeset -A slapdParams ## Variables read by readConf are placed here.
readConf $slapdDir/config/slapd.conf \
	port rootdn pw_storagescheme timelimit sizelimit rootpw

startSlapd

ldapReadBase ${slapdParams[port]:=389} ## sets namingContexts[], srvVLVsupport
ldapArgs="$ldapArgs -p ${slapdParams[port]}"

if [ -z "$nisDomain" ]; then
    # Not specified on the command line, therefore:
    getNisDomain ## Sets nisDomain
fi

if [ -z "$ldapRootdn" ]; then
    # Not specified on the command line, therefore:
    ldapRootdn=${slapdParams[rootdn]//\"/}
fi

hr
print "${bold}Password Authentication${norm}"
showInfo bindpw

while [ -z ${testPw} ]
do
    if [ -z "$ldapPw" ]; then
        getPassword "Enter \"$ldapRootdn\" password"
        ldapPw=$REPLY
     fi
     testPw=$($slapdDir/getpwenc SHA ${ldapPw})
     if [ ${testPw} != ${slapdParams[rootpw]} ]; then
	debug 3 "${testPw} ${slapdParams[rootpw]}"
	ldapPw=""
	testPw=""
 	print "Invalid Passowrd"
    fi
done
print "Password verified successfully"

##readConf $slapdDir/config/slapd.ldbm.conf \
##	lookthroughlimit cachesize dbcachesize

ldapAddAttr # Step 1

updateSlapdOcConf $slapdDir # Step 2

ldapAddObj # step 3

updatePasswdScheme # step 4

ldapAddTopClasses $namecontext $nisDomain # step 5

change_aci $namecontext	${slapdParams[port]} # step 6

ldapAddVLVaci # step 7

ldapAddProxyAgent # step 8, 9 and 10

ldapAddIndex # step 11

ldapAddVlv $namecontext # step 12

updateCache

msg 1 " Restarting slapd to ensure all changes have been filed."

stopSlapd
startSlapd


hr
msg 0 "Slapd configuration completed.  You will now need to import your data."
cat <<EOF 
A log of this setup can be found in "$logFile".

I recommend that you once again backup the Directory Server directory.
Don't overwrite your last backup!

EOF

exit 0
