#!/usr/bin/ksh
#
# Copyright (c) 2001-2002 by Sun Microsystems, Inc.
# All rights reserved.
#
# This script is for Starfire domains to detect main SSP in a dual SSP
# environment. It is designed to use on configurations where floating SSP 
# name and IP are either not valid or not supported. The script has been
# tested on SSP version 3.3/3.4/3.5.
#
# Installation:
#	Install on Starfire domains. Set up a cronjob to run this script
#  	periodically. The suggested frequency for running this script is
#   	every 3 minutes. Create file /etc/sspifs to contain IP addresses 
#       of two SSPs in the domain private subnet, followed by SSP host 
#	names. The address and the host	name should be separated by space 
#	only. Below is a sample file installed on domain_A who has private
#	interface 192.168.153.115, while 192.168.153.113 and 192.168.153.114 
#	are two private interfaces of SSP xf4-ssp and xf4-ssp2, respectively. 
#	These interfaces all belong to domain_A private subnet.
#
#		192.168.153.113 xf4-ssp
#		192.168.153.114 xf4-ssp2 	
#
# Usage:
#	This script is designed to run as a cronjob. No argument needed.
#   	A part of sample crontab setting is listed below:
#
#		0 * * * * /find_main_ssp.ksh > /dev/null 2>&1
#		3 * * * * /find_main_ssp.ksh > /dev/null 2>&1
#		6 * * * * /find_main_ssp.ksh > /dev/null 2>&1
#		9 * * * * /find_main_ssp.ksh > /dev/null 2>&1
#
#
# Some globals we use
#
RPCFILE=/tmp/rpc.result
SSPMAIN=/etc/ssphostname
SSPIFS=/etc/sspifs
TMP_SSPIFS=/tmp/sspifs
CONTOSTR="ready and waiting"
CONNOSTR="Program not registered"
CHANGE=

#
# Make sure both /etc/ssphostname and /etc/sspifs exist
#
if [[ ! -f ${SSPMAIN} ]]
then
   echo "File ${SSPMAIN} not present; exiting"
   exit 1
fi   

if [[ ! -f ${SSPIFS} ]]
then
   echo "File ${SSPIFS} not present; exiting"
   exit 1
fi   

#
# Preprocess file /etc/sspifs  
# 	- get rid of comments and empty lines 
# 	- move sspifs to temp file
# 	- make sure it only has two entries 
#
grep -v '#' ${SSPIFS} | grep '.' > ${TMP_SSPIFS}
if [[ ! -f ${TMP_SSPIFS} ]]
then
   echo "Can not create file ${TMP_SSPIFS}; exiting"
   exit 1
fi   

line=`wc -l ${TMP_SSPIFS}`
linenum=`echo $line | cut -d' ' -f1`
if [[ $linenum -ne 2 ]] 
then
   echo "File ${SSPIFS} should only have two entries; exiting"
   rm -rf ${TMP_SSPIFS}
   exit 1
fi 

#
# Find out whether current setting for ssphostname still valid 
#
main=`head -1 ${SSPMAIN} | cut -d' ' -f1` 
if ! ping $main 3 >/dev/null 2>&1
then
   # main SSP maybe in the process of reboot, wait for a while and retry
   echo "can not ping main ssp $main, sleep 120 sec"
   sleep 120 
fi
 
#
# I18N support
#
(LANG=C; LC_ALL=C; export LANG LC_ALL

#
# Find out whether need to change main SSP name
#
if ping $main 3 >/dev/null 2>&1
then
   main_if=`grep -w $main ${TMP_SSPIFS} | cut -d' ' -f1`
   spare_if=`grep -v -w $main ${TMP_SSPIFS} | cut -d' ' -f1`
   rpcinfo -s $main_if | grep -w 100254 > ${RPCFILE} 2>&1
   if [[ ! -f ${RPCFILE} ]]
   then
   	echo "Can not create file ${RPCFILE}; exiting"
   	rm -rf ${TMP_SSPIFS}
   	exit 1
   fi   

   if ! grep -w 100254 ${RPCFILE} >/dev/null 2>&1
   then
       	# either SSP daemon is not running, or SSP role changed 
       	CHANGE=true      
   else 
	# need to check spare SSP
       	rpcinfo -s $spare_if | grep -w 100254 > ${RPCFILE} 2>&1
       	if grep -w 100254 ${RPCFILE} >/dev/null 2>&1
       	then
            # both SSPs have rpc 100254 registered, need to conduct further test
            rpcinfo -T tcp $main_if 100254 > ${RPCFILE} 2>&1 
	    if grep "${CONNOSTR}" ${RPCFILE} >/dev/null 2>&1
	    then
		CHANGE=true
	    fi
       	fi
   fi
else
   # main SSP maybe down, need to find out whether spare SSP has taken over
   CHANGE=true
fi
 
#
# Find out what is the new main SSP now, only make changes when
# positive response is received from another SSP
#
if [[ "$CHANGE" == true ]]
then 
   cat ${TMP_SSPIFS} | while read line ; do
	ssp_if=`echo $line | cut -d' ' -f1`
	ssp=`echo $line | cut -d' ' -f2`
   	if [[ "$ssp" != "$main" ]]
   	then 
            if ping $ssp 3 >/dev/null 2>&1
            then
            	rpcinfo -s $ssp_if | grep -w 100254 > ${RPCFILE} 2>&1
            	if grep -w 100254 ${RPCFILE} >/dev/null 2>&1 
            	then
                    # update ssphostname file and restart cvcd
		    echo "change main ssp to $ssp and restart cvcd"
                    echo $ssp > ${SSPMAIN}
                    pkill -9 cvcd
                    sleep 3
                    /platform/SUNW,Ultra-Enterprise-10000/lib/cvcd
            	fi
            fi
   	fi
    done
fi

rm -rf ${RPCFILE}
rm -rf ${TMP_SSPIFS}
)


