/*
 *  asmtp.c - Part of AFD, an automatic file distribution program.
 *  Copyright (c) 2000 - 2005 Holger Kiehl <Holger.Kiehl@dwd.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "asmtpdefs.h"

DESCR__S_M1
/*
 ** NAME
 **   asmtp - send files via SMTP automaticaly
 **
 ** SYNOPSIS
 **   asmtp [options] [file 1 ... file n]
 **
 **   options
 **       --version               - Show current version
 **       -a <user@host>          - The address where the mail is sent.
 **       -b <block size>         - Transfer block size in bytes.
 **       -e                      - Encode files in BASE64.
 **       -f <filename file>      - List of filenames to send.
 **       -h <hostname|IP number> - Recipient hostname or IP of this mail.
 **       -i <from address>       - Send a from header.
 **       -o <reply-to address>   - Insert a reply-to header.
 **       -p <port number>        - Remote port number of SMTP-server.
 **       -m <mailserver-address> - Mailserver that will send this mail.
 **       -n                      - Filename is subject.
 **       -r                      - Remove transmitted file.
 **       -s <subject>            - Subject of this mail.
 **       -t <timout>             - SMTP timeout in seconds.
 **       -u <user>               - The user who should get the mail.
 **       -v                      - Verbose. Shows all SMTP commands
 **                                 and the reply from the remote server.
 **       -y                      - File name is user.
 **       -?                      - Display some help.
 **
 ** DESCRIPTION
 **   asmtp sends the given files to the defined recipient via SMTP
 **   It does so by using it's own SMTP-client.
 **
 ** RETURN VALUES
 **
 ** AUTHOR
 **   H.Kiehl
 **
 ** HISTORY
 **   24.12.2000 H.Kiehl Created
 **   04.08.2002 H.Kiehl Added To:, From: and Reply-To headers.
 **
 */
DESCR__E_M1

#include <stdio.h>                     /* fprintf(), sprintf()           */
#include <string.h>                    /* strcpy(), strcat(), strcmp(),  */
                                       /* strerror()                     */
#include <stdlib.h>                    /* malloc(), free()               */
#include <ctype.h>                     /* isdigit()                      */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>                    /* signal()                       */
#include <unistd.h>                    /* unlink(), close()              */
#include <errno.h>
#include "smtpdefs.h"
#include "version.h"

/* Global variables */
int                  line_length = 0,  /* encode_base64()                */
                     *no_of_listed_files,
                     sys_log_fd = STDERR_FILENO,
                     transfer_log_fd = STDERR_FILENO,
                     timeout_flag,
                     sigpipe_flag;
long                 transfer_timeout;
char                 msg_str[MAX_RET_MSG_LENGTH],
                     *p_work_dir = NULL;
struct data          db;
struct filename_list *rl;
const char           *sys_log_name = SYSTEM_LOG_FIFO;

/* Local functions */
static void          asmtp_exit(void),
                     sig_bus(int),
                     sig_segv(int),
                     sig_pipe(int),
                     sig_exit(int);


/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$ main() $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
int
main(int argc, char *argv[])
{
   int         blocksize,
               fd,
               j,
               status,
               loops,
               rest,
               files_send = 0,
               no_of_files_done = 0,
               write_size;
   size_t      length;
   off_t       file_size_done = 0,
               local_file_size,
               no_of_bytes;
   char        *buffer,
               *buffer_ptr,
               *encode_buffer,
               *file_ptr,
               host_name[256],
               local_user[MAX_FILENAME_LENGTH],
               multipart_boundary[MAX_FILENAME_LENGTH],
               remote_user[MAX_FILENAME_LENGTH],
               final_filename[MAX_FILENAME_LENGTH],
               *ptr,
               *smtp_buffer;
   struct stat stat_buf;

   CHECK_FOR_VERSION(argc, argv);

   /* Do some cleanups when we exit */
   if (atexit(asmtp_exit) != 0)
   {
      (void)rec(sys_log_fd, FATAL_SIGN,
                "Could not register exit function : %s (%s %d)\n",
                strerror(errno), __FILE__, __LINE__);
      exit(INCORRECT);
   }
   if ((signal(SIGINT, sig_exit) == SIG_ERR) ||
       (signal(SIGSEGV, sig_segv) == SIG_ERR) ||
       (signal(SIGBUS, sig_bus) == SIG_ERR) ||
       (signal(SIGHUP, SIG_IGN) == SIG_ERR) ||
       (signal(SIGPIPE, sig_pipe) == SIG_ERR))
   {
      (void)rec(sys_log_fd, FATAL_SIGN, "signal() error : %s (%s %d)\n",
                strerror(errno), __FILE__, __LINE__);
      exit(INCORRECT);
   }

   /* Initialise variables */
   init_asmtp(argc, argv, &db);
   msg_str[0] = '\0';
   blocksize = db.blocksize;

   /* Set SMTP timeout value */
   transfer_timeout = db.transfer_timeout;

   if (db.smtp_server[0] == '\0')
   {
      (void)strcpy(db.smtp_server, SMTP_HOST_NAME);
   }

   /*
    * The extra buffer is needed to convert LF's to CRLF.
    */
   if ((smtp_buffer = malloc(((blocksize * 2) + 1))) == NULL)
   {
      (void)rec(sys_log_fd, ERROR_SIGN, "malloc() error : %s (%s %d)\n",
                strerror(errno), __FILE__, __LINE__);
      exit(ALLOC_ERROR);
   }

   sigpipe_flag = timeout_flag = OFF;

   /* Connect to remote SMTP-server */
   if ((status = smtp_connect(db.smtp_server, db.port)) != SUCCESS)
   {
      trans_log(ERROR_SIGN, __FILE__, __LINE__, msg_str,
                "SMTP connection to <%s> at port %d failed (%d).",
                db.smtp_server, db.port, status);
      exit(eval_timeout(CONNECT_ERROR));
   }
   else
   {
      if (db.verbose == YES)
      {
         trans_log(INFO_SIGN, __FILE__, __LINE__, msg_str, "Connected.");
      }
   }

   /* Now send HELO */
   if (gethostname(host_name, 255) < 0)
   {
      (void)rec(sys_log_fd, ERROR_SIGN, "gethostname() error : %s (%s %d)\n",
                strerror(errno), __FILE__, __LINE__);
      exit(INCORRECT);
   }
   if ((status = smtp_helo(host_name)) != SUCCESS)
   {
      trans_log(ERROR_SIGN, __FILE__, __LINE__, msg_str,
                "Failed to send HELO to <%s> (%d).", db.smtp_server, status);
      (void)smtp_quit();
      exit(eval_timeout(CONNECT_ERROR));
   }
   else
   {
      if (db.verbose == YES)
      {
         trans_log(INFO_SIGN, __FILE__, __LINE__, msg_str, "Send HELO.");
      }
   }

   /* Prepare local and remote user name */
   if ((ptr = getenv("LOGNAME")) != NULL)
   {
      (void)sprintf(local_user, "%s@%s", ptr, host_name);
   }
   else
   {
      (void)sprintf(local_user, "%s@%s", AFD_USER_NAME, host_name);
   }

   if ((db.flag & FILE_NAME_IS_USER) == 0)
   {
      (void)sprintf(remote_user, "%s@%s", db.user, db.hostname);
   }

   /* Allocate buffer to read data from the source file. */
   if ((buffer = malloc(blocksize + 4)) == NULL)
   {
      (void)rec(sys_log_fd, ERROR_SIGN, "malloc() error : %s (%s %d)\n",
                strerror(errno), __FILE__, __LINE__);
      exit(ALLOC_ERROR);
   }

   if (db.flag & ATTACH_FILE)
   {
      if ((encode_buffer = malloc((2 * (blocksize + 1)) + 1)) == NULL)
      {
         (void)rec(sys_log_fd, ERROR_SIGN, "malloc() error : %s (%s %d)\n",
                   strerror(errno), __FILE__, __LINE__);
         exit(ALLOC_ERROR);
      }

      /*
       * When encoding in base64 is done the blocksize must be
       * divideable by three!!!!
       */
      blocksize = blocksize - (blocksize % 3);
   }

   multipart_boundary[0] = '\0';

   /* Send all files */
   for (files_send = 0; files_send < db.no_of_files; files_send++)
   {
      if ((db.realname != NULL) && (db.realname[files_send][0] != '\0'))
      {
         file_ptr = db.realname[files_send];
      }
      else
      {
         file_ptr = db.filename[files_send];
         length = strlen(file_ptr);
         while (length != 0)
         {
            if (db.filename[files_send][length - 1] == '/')
            {
               file_ptr = &db.filename[files_send][length];
               break;
            }
            length--;
         }
      }
      (void)strcpy(final_filename, file_ptr);

      /* Send local user name */
      if ((status = smtp_user(local_user)) != SUCCESS)
      {
         trans_log(ERROR_SIGN, __FILE__, __LINE__, msg_str,
                   "Failed to send local user <%s> (%d).", local_user, status);
         (void)smtp_quit();
         exit(eval_timeout(USER_ERROR));
      }
      else
      {
         if (db.verbose == YES)
         {
            trans_log(INFO_SIGN, __FILE__, __LINE__, msg_str,
                      "Entered local user name %s.", local_user);
         }
      }

      /* Send remote user name */
      if ((status = smtp_rcpt(remote_user)) != SUCCESS)
      {
         trans_log(ERROR_SIGN, __FILE__, __LINE__, msg_str,
                   "Failed to send remote user <%s> (%d).", remote_user, status);
         (void)smtp_quit();
         exit(eval_timeout(REMOTE_USER_ERROR));
      }
      else
      {
         if (db.verbose == YES)
         {
            trans_log(INFO_SIGN, __FILE__, __LINE__, msg_str,
                      "Remote address %s accepted by SMTP-server.",
                      remote_user);
         }
      }

      /* Enter data mode */
      if ((status = smtp_open()) != SUCCESS)
      {                                     
         trans_log(ERROR_SIGN, __FILE__, __LINE__, msg_str,
                   "Failed to set DATA mode (%d).", status);
         (void)smtp_quit();                                 
         exit(eval_timeout(DATA_ERROR));
      }                   
      else
      {   
         if (db.verbose == YES)
         {                           
            trans_log(INFO_SIGN, __FILE__, __LINE__, msg_str, "Set DATA mode.");
         }                                                                
      }

      /* Open local file */
      if ((fd = open(db.filename[files_send], O_RDONLY)) < 0)
      {
         if (db.verbose == YES)
         {
            trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                      "Failed to open() local file %s",
                      db.filename[files_send]);
         }

         /* Close remote file. */
         if ((status = smtp_close()) != SUCCESS)
         {
            WHAT_DONE(file_size_done, no_of_files_done);
            trans_log(ERROR_SIGN, __FILE__, __LINE__, msg_str,
                      "Failed to close data mode (%d).", status);
            (void)smtp_quit();
            exit(eval_timeout(CLOSE_REMOTE_ERROR));
         }
         else
         {
            if (db.verbose == YES)
            {
               trans_log(INFO_SIGN, __FILE__, __LINE__, msg_str,
                         "Closing data mode.");
            }
         }
         continue;
      }
      if (fstat(fd, &stat_buf) == -1)
      {
         if (db.verbose == YES)
         {
            trans_log(INFO_SIGN, __FILE__, __LINE__, NULL,
                      "Failed to fstat() local file %s",
                      db.filename[files_send]);
         }
         WHAT_DONE(file_size_done, no_of_files_done);
         (void)smtp_close();
         (void)smtp_quit();
         exit(STAT_ERROR);
      }
      else
      {
         if (!S_ISREG(stat_buf.st_mode))
         {
            if (db.verbose == YES)
            {
               trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                         "Local file %s is not a regular file.",
                         db.filename[files_send]);
            }

            /* Close remote file. */
            if ((status = smtp_close()) != SUCCESS)
            {
               WHAT_DONE(file_size_done, no_of_files_done);
               trans_log(ERROR_SIGN, __FILE__, __LINE__, msg_str,
                         "Failed to close data mode (%d).", status);
               (void)smtp_quit();
               exit(eval_timeout(CLOSE_REMOTE_ERROR));
            }
            else
            {
               if (db.verbose == YES)
               {
                  trans_log(INFO_SIGN, __FILE__, __LINE__, msg_str,
                            "Closing data mode.");
               }
            }
            (void)close(fd);
            continue;
         }
      }
      local_file_size = stat_buf.st_size;
      if (db.verbose == YES)
      {
         trans_log(INFO_SIGN, __FILE__, __LINE__, NULL,
                   "Open local file %s", db.filename[files_send]);
      }

      /* Read (local) and write (remote) file */
      no_of_bytes = 0;
      loops = local_file_size / blocksize;
      rest = local_file_size % blocksize;

      if (db.from != NULL)
      {
         length = sprintf(buffer, "From: %s\n", db.from);
         if (smtp_write(buffer, NULL, length) < 0)
         {
            WHAT_DONE(file_size_done, no_of_files_done);
            trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                      "Failed to write From to SMTP-server.");
            (void)smtp_quit();
            exit(eval_timeout(WRITE_REMOTE_ERROR));
         }
         no_of_bytes = length;
      }
      if (db.reply_to != NULL)
      {
         length = sprintf(buffer, "Reply-To: %s\n", db.reply_to);
         if (smtp_write(buffer, NULL, length) < 0)
         {
            WHAT_DONE(file_size_done, no_of_files_done);
            trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                      "Failed to write Reply-To to SMTP-server.");
            (void)smtp_quit();
            exit(eval_timeout(WRITE_REMOTE_ERROR));
         }
         no_of_bytes += length;
      }

      if (db.subject != NULL)
      {
         length = sprintf(buffer, "Subject : %s\r\n", db.subject);
         if (smtp_write(buffer, NULL, length) < 0)
         {
            WHAT_DONE(file_size_done, no_of_files_done);
            trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                      "Failed to write subject to SMTP-server.");
            (void)smtp_quit();
            exit(eval_timeout(WRITE_REMOTE_ERROR));
         }
         no_of_bytes += length;
      }
      else if (db.flag & FILE_NAME_IS_SUBJECT)
           {
              length = sprintf(buffer, "Subject : %s\r\n", final_filename);
              if (smtp_write(buffer, NULL, length) < 0)
              {
                 WHAT_DONE(file_size_done, no_of_files_done);
                 trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                           "Failed to write the filename as subject to SMTP-server.");
                 (void)smtp_quit();
                 exit(eval_timeout(WRITE_REMOTE_ERROR));
              }
              no_of_bytes += length;
           } /* if (db.flag & FILE_NAME_IS_SUBJECT) */

      length = sprintf(buffer, "To: %s\r\n", remote_user);
      if (smtp_write(buffer, NULL, length) < 0)
      {
         WHAT_DONE(file_size_done, no_of_files_done);
         trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                   "Failed to write To header to SMTP-server.");
         (void)smtp_quit();
         exit(eval_timeout(WRITE_REMOTE_ERROR));
      }
      no_of_bytes += length;

      /* Send MIME information. */
      if (db.flag & ATTACH_FILE)
      {
         size_t length;

         if (multipart_boundary[0] != '\0')
         {
            length = sprintf(buffer,
                             "MIME-Version: 1.0 (produced by AFD %s)\r\nContent-Type: MULTIPART/MIXED; BOUNDARY=\"%s\"\r\n",
                             PACKAGE_VERSION, multipart_boundary);
            buffer_ptr = buffer;
         }
         else
         {
            length = sprintf(encode_buffer,
                             "MIME-Version: 1.0 (produced by AFD %s)\r\nContent-Type: APPLICATION/octet-stream; name=\"%s\"\r\nContent-Transfer-Encoding: BASE64\r\n\r\n",
                             PACKAGE_VERSION, final_filename);
            buffer_ptr = encode_buffer;
         }

         if (smtp_write(buffer_ptr, NULL, length) < 0)
         {
            WHAT_DONE(file_size_done, no_of_files_done);
            trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                      "Failed to write start of multipart boundary to SMTP-server.");
            (void)smtp_quit();
            exit(eval_timeout(WRITE_REMOTE_ERROR));
         }
         no_of_bytes += length;
      } /* if (db.flag & ATTACH_FILE) */
      else
      {
         if (smtp_buffer != NULL)
         {
            smtp_buffer[0] = 0;
         }
      }
      if (smtp_write("\r\n", NULL, 2) < 0)
      {
         WHAT_DONE(file_size_done, no_of_files_done);
         trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                   "Failed to write carriage return line feed to mark end of header to SMTP-server.");
         (void)smtp_quit();
         exit(eval_timeout(WRITE_REMOTE_ERROR));
      }
      no_of_bytes += 2;

      for (;;)
      {
         for (j = 0; j < loops; j++)
         {
            if (read(fd, buffer, blocksize) != blocksize)
            {
               WHAT_DONE(file_size_done, no_of_files_done);
               trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                         "Failed to read() %s : %s",
                         db.filename[files_send], strerror(errno));
               (void)smtp_close();
               (void)smtp_quit();
               exit(READ_LOCAL_ERROR);
            }
            if (db.flag & ATTACH_FILE)
            {
               write_size = encode_base64((unsigned char *)buffer, blocksize,
                                          (unsigned char *)encode_buffer);
               if (smtp_write(encode_buffer, NULL, write_size) < 0)
               {
                  WHAT_DONE(file_size_done, no_of_files_done);
                  trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                            "Failed to write data from the source file to the SMTP-server.");
                  (void)smtp_quit();
                  exit(eval_timeout(WRITE_REMOTE_ERROR));
               }
            }
            else
            {
               if (smtp_write(buffer, smtp_buffer, blocksize) < 0)
               {
                  WHAT_DONE(file_size_done, no_of_files_done);
                  trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                            "Failed to write data from the source file to the SMTP-server.");
                  (void)smtp_quit();
                  exit(eval_timeout(WRITE_REMOTE_ERROR));
               }
               write_size = blocksize;
            }
            file_size_done += write_size;
            no_of_bytes += write_size;
         }

         if (rest > 0)
         {
            if (read(fd, buffer, rest) != rest)
            {
               WHAT_DONE(file_size_done, no_of_files_done);
               trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                         "Failed to read() rest from %s : %s",
                         db.filename[files_send], strerror(errno));
               (void)smtp_close();
               (void)smtp_quit();
               exit(READ_LOCAL_ERROR);
            }
            if (db.flag & ATTACH_FILE)
            {
               write_size = encode_base64((unsigned char *)buffer, rest,
                                          (unsigned char *)encode_buffer);
               if (smtp_write(encode_buffer, NULL, write_size) < 0)
               {
                  WHAT_DONE(file_size_done, no_of_files_done);
                  trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                            "Failed to write the rest data from the source file to the SMTP-server.");
                  (void)smtp_quit();
                  exit(eval_timeout(WRITE_REMOTE_ERROR));
               }
            }
            else
            {
               if (smtp_write(buffer, smtp_buffer, rest) < 0)
               {
                  WHAT_DONE(file_size_done, no_of_files_done);
                  trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                            "Failed to write the rest data from the source file to the SMTP-server.");
                  (void)smtp_quit();
                  exit(eval_timeout(WRITE_REMOTE_ERROR));
               }
               write_size = rest;
            }
            file_size_done += write_size;
            no_of_bytes += write_size;
         } /* if (rest > 0) */

         /*
          * Since there are always some users sending files to the
          * AFD not in dot notation, lets check here if this is really
          * the EOF.
          * If not lets continue so long until we hopefully have reached
          * the EOF.
          * NOTE: This is NOT a fool proof way. There must be a better
          *       way!
          */
         if (stat(db.filename[files_send], &stat_buf) == 0)
         {
            if (stat_buf.st_size > local_file_size)
            {
               loops = (stat_buf.st_size - local_file_size) / blocksize;
               rest = (stat_buf.st_size - local_file_size) % blocksize;
               local_file_size = stat_buf.st_size;

               /*
                * Give a warning in the system log, so some action
                * can be taken against the originator.
                */
               (void)rec(sys_log_fd, WARN_SIGN,
                         "Someone is still writting to file %s. (%s %d)\n",
                         db.filename[files_send], __FILE__, __LINE__);
            }
            else
            {
               break;
            }
         }
         else
         {
            break;
         }
      } /* for (;;) */

      /* Write boundary end if neccessary. */
      if (db.flag & ATTACH_FILE)
      {
         size_t length;

         /* Write boundary */
         length = sprintf(buffer, "\r\n--%s--\r\n", multipart_boundary);

         if (smtp_write(buffer, NULL, length) < 0)
         {
            WHAT_DONE(file_size_done, no_of_files_done);
            trans_log(ERROR_SIGN, __FILE__, __LINE__, NULL,
                      "Failed to write end of multipart boundary to SMTP-server.");
            (void)smtp_quit();
            exit(eval_timeout(WRITE_REMOTE_ERROR));
         }
         no_of_bytes += length;
      }

      /* Close local file. */
      if (close(fd) < 0)
      {
         trans_log(WARN_SIGN, __FILE__, __LINE__, NULL,
                   "Failed to close() local file %s : %s",
                   final_filename, strerror(errno));

         /*
          * Hmm. Lets hope this does not happen to offen. Else
          * we will have too many file descriptors open.
          */
      }

      /* Close remote file. */
      if ((status = smtp_close()) != SUCCESS)
      {
         WHAT_DONE(file_size_done, no_of_files_done);
         trans_log(ERROR_SIGN, __FILE__, __LINE__, msg_str,
                   "Failed to close data mode (%d).", status);
         (void)smtp_quit();
         exit(eval_timeout(CLOSE_REMOTE_ERROR));
      }
      else
      {
         if (db.verbose == YES)
         {
            trans_log(INFO_SIGN, __FILE__, __LINE__, msg_str,
                      "Closing data mode.");
         }
      }

      no_of_files_done++;

      if (db.remove == YES)
      {
         /* Delete the file we just have send */
         if (unlink(db.filename[files_send]) < 0)
         {
            (void)rec(sys_log_fd, ERROR_SIGN,
                      "Could not unlink() local file %s after sending it successfully : %s (%s %d)\n",
                      strerror(errno), db.filename[files_send],
                      __FILE__, __LINE__);
         }
      }
   } /* for (files_send = 0; files_send < db.no_of_files; files_send++) */

   WHAT_DONE(file_size_done, no_of_files_done);
   msg_str[0] = '\0';
   free(buffer);

   /* Logout again */
   if ((status = smtp_quit()) != SUCCESS)
   {
      trans_log(WARN_SIGN, __FILE__, __LINE__, msg_str,
                "Failed to disconnect from SMTP-server (%d).", status);
   }
   else
   {
      if (db.verbose == YES)
      {
         trans_log(INFO_SIGN, __FILE__, __LINE__, msg_str, "Logged out.");
      }
   }

   /* Don't need the ASCII buffer */
   if (db.flag == ATTACH_FILE)
   {
      free(encode_buffer);
   }

   exit(SUCCESS);
}


/*++++++++++++++++++++++++++++++ asmtp_exit() ++++++++++++++++++++++++++++*/
static void
asmtp_exit(void)
{
   if (db.filename != NULL)
   {
      FREE_RT_ARRAY(db.filename);
   }
   if (db.realname != NULL)
   {
      FREE_RT_ARRAY(db.realname);
   }
   return;
}


/*++++++++++++++++++++++++++++++ sig_pipe() +++++++++++++++++++++++++++++*/
static void
sig_pipe(int signo)
{
   /* Ignore any future signals of this kind. */
   if (signal(SIGPIPE, SIG_IGN) == SIG_ERR)
   {
      (void)rec(sys_log_fd, ERROR_SIGN, "signal() error : %s (%s %d)\n",
                strerror(errno), __FILE__, __LINE__);
   }
   sigpipe_flag = ON;

   return;
}


/*++++++++++++++++++++++++++++++ sig_segv() +++++++++++++++++++++++++++++*/
static void
sig_segv(int signo)
{
   (void)rec(sys_log_fd, DEBUG_SIGN,
             "Aaarrrggh! Received SIGSEGV. Remove the programmer who wrote this! (%s %d)\n",
             __FILE__, __LINE__);
   exit(INCORRECT);
}


/*++++++++++++++++++++++++++++++ sig_bus() ++++++++++++++++++++++++++++++*/
static void
sig_bus(int signo)
{
   (void)rec(sys_log_fd, DEBUG_SIGN,
             "Uuurrrggh! Received SIGBUS. (%s %d)\n",
             __FILE__, __LINE__);
   exit(INCORRECT);
}


/*++++++++++++++++++++++++++++++ sig_exit() +++++++++++++++++++++++++++++*/
static void
sig_exit(int signo)
{
   exit(INCORRECT);
}
